<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://plugins.quickpr.co/
 * @since      1.0.0
 *
 * @package    Quick_Sitemap
 * @subpackage Quick_Sitemap/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for
 * the public-facing side of the site.
 *
 * @package    Quick_Sitemap
 * @subpackage Quick_Sitemap/public
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Sitemap_Public {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * The options of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $options    The options of this plugin.
     */
    private $options;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of the plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->options = get_option('quick_sitemap_options', array());
    }

    /**
     * Add rewrite rules for sitemaps.
     *
     * @since    1.0.0
     */
    public function add_sitemap_rewrite_rules() {
        // Check if sitemap is enabled
        if (!isset($this->options['sitemap_enabled']) || !$this->options['sitemap_enabled']) {
            return;
        }

        // Disable WordPress built-in sitemap
        add_filter('wp_sitemaps_enabled', '__return_false');
        
        // Main sitemap index
        add_rewrite_rule(
            '^sitemap\.xml$',
            'index.php?quick_sitemap=index',
            'top'
        );

        // Sitemap XSL stylesheet
        add_rewrite_rule(
            '^sitemap\.xsl$',
            'index.php?quick_sitemap=stylesheet',
            'top'
        );

        // Post type sitemaps
        $post_types = get_post_types(array('public' => true), 'names');
        foreach ($post_types as $post_type) {
            add_rewrite_rule(
                '^sitemap-' . $post_type . '\.xml$',
                'index.php?quick_sitemap=post_type&quick_sitemap_post_type=' . $post_type,
                'top'
            );
        }

        // Taxonomy sitemaps
        $taxonomies = get_taxonomies(array('public' => true), 'names');
        foreach ($taxonomies as $taxonomy) {
            add_rewrite_rule(
                '^sitemap-tax-' . $taxonomy . '\.xml$',
                'index.php?quick_sitemap=taxonomy&quick_sitemap_taxonomy=' . $taxonomy,
                'top'
            );
        }

        // Author sitemap
        add_rewrite_rule(
            '^sitemap-author\.xml$',
            'index.php?quick_sitemap=author',
            'top'
        );

        // Image sitemap
        add_rewrite_rule(
            '^sitemap-images\.xml$',
            'index.php?quick_sitemap=images',
            'top'
        );

        // Video sitemap
        add_rewrite_rule(
            '^sitemap-videos\.xml$',
            'index.php?quick_sitemap=videos',
            'top'
        );

        // Google News sitemap
        add_rewrite_rule(
            '^sitemap-news\.xml$',
            'index.php?quick_sitemap=news',
            'top'
        );
        
        // Alternate URL for Google News sitemap (for compatibility)
        add_rewrite_rule(
            '^news-sitemap\.xml$',
            'index.php?quick_sitemap=news',
            'top'
        );

        // Add query vars
        add_filter('query_vars', function($vars) {
            $vars[] = 'quick_sitemap';
            $vars[] = 'quick_sitemap_post_type';
            $vars[] = 'quick_sitemap_taxonomy';
            return $vars;
        });
    }

    /**
     * Handle sitemap requests.
     *
     * @since    1.0.0
     */
    public function handle_sitemap_requests() {
        global $wp_query;

        // Check if this is a sitemap request
        if (!isset($wp_query->query_vars['quick_sitemap'])) {
            return;
        }

        // Check if sitemap is enabled
        if (!isset($this->options['sitemap_enabled']) || !$this->options['sitemap_enabled']) {
            wp_die(__('Sitemap is disabled.', 'quick-sitemap'));
        }

        // Get the sitemap type
        $sitemap_type = $wp_query->query_vars['quick_sitemap'];

        // Set the content type header
        if ($sitemap_type === 'stylesheet') {
            header('Content-Type: text/xsl; charset=UTF-8');
        } else {
            header('Content-Type: application/xml; charset=UTF-8');
        }

        // Disable caching if debugging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            header('Cache-Control: no-cache, no-store, must-revalidate');
            header('Pragma: no-cache');
            header('Expires: 0');
        } else {
            // Cache for 1 hour
            header('Cache-Control: max-age=3600');
        }

        // Create sitemap generator
        $generator = new Quick_Sitemap_Generator();

        // Generate the appropriate sitemap
        switch ($sitemap_type) {
            case 'index':
                $output = $generator->generate_sitemap_index();
                break;

            case 'post_type':
                $post_type = get_query_var('quick_sitemap_post_type');
                if (!$post_type || !post_type_exists($post_type)) {
                    wp_die(__('Invalid post type.', 'quick-sitemap'));
                }
                $output = $generator->generate_post_type_sitemap($post_type);
                break;

            case 'taxonomy':
                $taxonomy = get_query_var('quick_sitemap_taxonomy');
                if (!$taxonomy || !taxonomy_exists($taxonomy)) {
                    wp_die(__('Invalid taxonomy.', 'quick-sitemap'));
                }
                $output = $generator->generate_taxonomy_sitemap($taxonomy);
                break;

            case 'author':
                $output = $generator->generate_author_sitemap();
                break;

            case 'images':
                if (!isset($this->options['sitemap_include_images']) || !$this->options['sitemap_include_images']) {
                    wp_die(__('Image sitemap is disabled.', 'quick-sitemap'));
                }
                $output = $generator->generate_image_sitemap();
                break;

            case 'videos':
                if (!isset($this->options['sitemap_include_videos']) || !$this->options['sitemap_include_videos']) {
                    wp_die(__('Video sitemap is disabled.', 'quick-sitemap'));
                }
                $output = $generator->generate_video_sitemap();
                break;

            case 'news':
                if (!isset($this->options['sitemap_google_news']) || !$this->options['sitemap_google_news']) {
                    wp_die(__('Google News sitemap is disabled.', 'quick-sitemap'));
                }
                $output = $generator->generate_news_sitemap();
                // Update last generated time
                update_option('quick_sitemap_last_generated', time());
                break;

            case 'stylesheet':
                $output = $this->get_sitemap_stylesheet();
                break;

            default:
                wp_die(__('Invalid sitemap type.', 'quick-sitemap'));
        }

        // Output the sitemap
        echo $output;
        exit;
    }

    /**
     * Get the sitemap XSL stylesheet.
     *
     * @since    1.0.0
     * @return   string    The XSL stylesheet.
     */
    private function get_sitemap_stylesheet() {
        ob_start();
        include plugin_dir_path(__FILE__) . 'partials/quick-sitemap-stylesheet.xsl.php';
        return ob_get_clean();
    }

    /**
     * Ping search engines when content is updated.
     *
     * @since    1.0.0
     * @param    int       $post_id    The post ID.
     * @param    WP_Post   $post       The post object.
     * @param    bool      $update     Whether this is an existing post being updated.
     */
    public function ping_search_engines_on_update($post_id, $post, $update) {
        // Check if pinging is enabled
        if (!isset($this->options['sitemap_ping_search_engines']) || !$this->options['sitemap_ping_search_engines']) {
            return;
        }

        // Check if this is a revision or auto-save
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
            return;
        }

        // Check if post is published
        if ($post->post_status !== 'publish') {
            return;
        }

        // Check if post type is enabled in sitemap
        if (!isset($this->options['sitemap_items'][$post->post_type]) || !$this->options['sitemap_items'][$post->post_type]) {
            return;
        }

        // Get sitemap URL
        $sitemap_url = home_url('/sitemap.xml');

        // Ping Google
        if (isset($this->options['sitemap_ping_services']['google']) && $this->options['sitemap_ping_services']['google']) {
            wp_remote_get('https://www.google.com/ping?sitemap=' . urlencode($sitemap_url), array('timeout' => 5));
        }

        // Ping Bing
        if (isset($this->options['sitemap_ping_services']['bing']) && $this->options['sitemap_ping_services']['bing']) {
            wp_remote_get('https://www.bing.com/ping?sitemap=' . urlencode($sitemap_url), array('timeout' => 5));
        }
    }
}
