<?php
/**
 * The sitemap generator functionality of the plugin.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Quick_Sitemap
 * @subpackage Quick_Sitemap/includes
 */

/**
 * The sitemap generator class.
 *
 * This class handles the generation of XML sitemaps, including specialized
 * sitemaps for images, videos, and Google News.
 *
 * @since      1.0.0
 * @package    Quick_Sitemap
 * @subpackage Quick_Sitemap/includes
 * @author     Quick Plugins <info@example.com>
 */
class Quick_Sitemap_Generator {

    /**
     * The plugin options.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $options    The plugin options.
     */
    private $options;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     */
    public function __construct() {
        $this->options = get_option('quick_sitemap_options', array());
    }

    /**
     * Generate the main sitemap index.
     *
     * @since    1.0.0
     * @return   string    The XML content of the sitemap index.
     */
    public function generate_sitemap_index() {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        
        // Add stylesheet if enabled
        if (isset($this->options['sitemap_stylesheet']) && $this->options['sitemap_stylesheet']) {
            $xml .= '<?xml-stylesheet type="text/xsl" href="' . esc_url(home_url('/sitemap.xsl')) . '"?>';
        }
        
        $xml .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
        
        // Add post types sitemaps
        $post_types = $this->get_enabled_post_types();
        foreach ($post_types as $post_type) {
            $xml .= $this->get_sitemap_index_entry($post_type);
        }
        
        // Add taxonomy sitemaps
        $taxonomies = $this->get_enabled_taxonomies();
        foreach ($taxonomies as $taxonomy) {
            $xml .= $this->get_sitemap_index_entry('tax-' . $taxonomy);
        }
        
        // Add author sitemap if any posts exist
        if ($this->has_authors_with_posts()) {
            $xml .= $this->get_sitemap_index_entry('author');
        }
        
        // Add image sitemap if enabled
        if (isset($this->options['sitemap_include_images']) && $this->options['sitemap_include_images']) {
            $xml .= $this->get_sitemap_index_entry('images');
        }
        
        // Add video sitemap if enabled
        if (isset($this->options['sitemap_include_videos']) && $this->options['sitemap_include_videos']) {
            $xml .= $this->get_sitemap_index_entry('videos');
        }
        
        // Add Google News sitemap if enabled
        if (isset($this->options['sitemap_google_news']) && $this->options['sitemap_google_news']) {
            $xml .= $this->get_sitemap_index_entry('news');
        }
        
        $xml .= '</sitemapindex>';
        
        return $xml;
    }
    
    /**
     * Get sitemap index entry.
     *
     * @since    1.0.0
     * @param    string    $name    The name of the sitemap.
     * @return   string             The XML for the sitemap index entry.
     */
    private function get_sitemap_index_entry($name) {
        $url = home_url('/sitemap-' . $name . '.xml');
        $lastmod = $this->get_last_modified_date($name);
        
        $entry = '<sitemap>';
        $entry .= '<loc>' . esc_url($url) . '</loc>';
        
        if ($lastmod) {
            $entry .= '<lastmod>' . esc_html($lastmod) . '</lastmod>';
        }
        
        $entry .= '</sitemap>';
        
        return $entry;
    }
    
    /**
     * Get the last modified date for a sitemap.
     *
     * @since    1.0.0
     * @param    string    $name    The name of the sitemap.
     * @return   string             The last modified date in W3C format.
     */
    private function get_last_modified_date($name) {
        global $wpdb;
        
        // For post types
        if (post_type_exists($name)) {
            $latest_post = $wpdb->get_var($wpdb->prepare(
                "SELECT post_modified FROM $wpdb->posts WHERE post_type = %s AND post_status = 'publish' ORDER BY post_modified DESC LIMIT 1",
                $name
            ));
            
            if ($latest_post) {
                return mysql2date('c', $latest_post);
            }
        }
        
        // For taxonomies
        if (strpos($name, 'tax-') === 0) {
            $taxonomy = str_replace('tax-', '', $name);
            
            if (taxonomy_exists($taxonomy)) {
                $latest_term = $wpdb->get_var($wpdb->prepare(
                    "SELECT MAX(t.term_id) FROM $wpdb->terms t INNER JOIN $wpdb->term_taxonomy tt ON t.term_id = tt.term_id WHERE tt.taxonomy = %s",
                    $taxonomy
                ));
                
                if ($latest_term) {
                    // Get the latest post in this term
                    $latest_post = $wpdb->get_var($wpdb->prepare(
                        "SELECT MAX(p.post_modified) FROM $wpdb->posts p INNER JOIN $wpdb->term_relationships tr ON p.ID = tr.object_id WHERE tr.term_taxonomy_id = %d AND p.post_status = 'publish'",
                        $latest_term
                    ));
                    
                    if ($latest_post) {
                        return mysql2date('c', $latest_post);
                    }
                }
            }
        }
        
        // For authors
        if ($name === 'author') {
            $latest_post = $wpdb->get_var(
                "SELECT MAX(post_modified) FROM $wpdb->posts WHERE post_status = 'publish'"
            );
            
            if ($latest_post) {
                return mysql2date('c', $latest_post);
            }
        }
        
        // Default to current time
        return current_time('c');
    }
    
    /**
     * Generate a sitemap for a specific post type.
     *
     * @since    1.0.0
     * @param    string    $post_type    The post type to generate a sitemap for.
     * @return   string                  The XML content of the sitemap.
     */
    public function generate_post_type_sitemap($post_type) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        
        // Add stylesheet if enabled
        if (isset($this->options['sitemap_stylesheet']) && $this->options['sitemap_stylesheet']) {
            $xml .= '<?xml-stylesheet type="text/xsl" href="' . esc_url(home_url('/sitemap.xsl')) . '"?>';
        }
        
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"';
        $xml .= ' xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"';
        $xml .= ' xsi:schemaLocation="http://www.sitemaps.org/schemas/sitemap/0.9 http://www.sitemaps.org/schemas/sitemap/0.9/sitemap.xsd"';
        
        // Add image namespace if including images
        if (isset($this->options['sitemap_include_images']) && $this->options['sitemap_include_images']) {
            $xml .= ' xmlns:image="http://www.google.com/schemas/sitemap-image/1.1"';
        }
        
        // Add video namespace if including videos
        if (isset($this->options['sitemap_include_videos']) && $this->options['sitemap_include_videos']) {
            $xml .= ' xmlns:video="http://www.google.com/schemas/sitemap-video/1.1"';
        }
        
        // Add language namespace if multilingual
        if ($this->is_multilingual_site()) {
            $xml .= ' xmlns:xhtml="http://www.w3.org/1999/xhtml"';
        }
        
        $xml .= '>';
        
        // Get posts for this post type
        $posts = $this->get_posts_for_sitemap($post_type);
        
        foreach ($posts as $post) {
            $xml .= $this->get_url_entry_for_post($post);
        }
        
        $xml .= '</urlset>';
        
        return $xml;
    }
    
    /**
     * Generate a sitemap for a specific taxonomy.
     *
     * @since    1.0.0
     * @param    string    $taxonomy    The taxonomy to generate a sitemap for.
     * @return   string                 The XML content of the sitemap.
     */
    public function generate_taxonomy_sitemap($taxonomy) {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        
        // Add stylesheet if enabled
        if (isset($this->options['sitemap_stylesheet']) && $this->options['sitemap_stylesheet']) {
            $xml .= '<?xml-stylesheet type="text/xsl" href="' . esc_url(home_url('/sitemap.xsl')) . '"?>';
        }
        
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"';
        $xml .= ' xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"';
        $xml .= ' xsi:schemaLocation="http://www.sitemaps.org/schemas/sitemap/0.9 http://www.sitemaps.org/schemas/sitemap/0.9/sitemap.xsd"';
        
        // Add language namespace if multilingual
        if ($this->is_multilingual_site()) {
            $xml .= ' xmlns:xhtml="http://www.w3.org/1999/xhtml"';
        }
        
        $xml .= '>';
        
        // Get terms for this taxonomy
        $terms = $this->get_terms_for_sitemap($taxonomy);
        
        foreach ($terms as $term) {
            $xml .= $this->get_url_entry_for_term($term, $taxonomy);
        }
        
        $xml .= '</urlset>';
        
        return $xml;
    }
    
    /**
     * Generate a sitemap for authors.
     *
     * @since    1.0.0
     * @return   string    The XML content of the sitemap.
     */
    public function generate_author_sitemap() {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        
        // Add stylesheet if enabled
        if (isset($this->options['sitemap_stylesheet']) && $this->options['sitemap_stylesheet']) {
            $xml .= '<?xml-stylesheet type="text/xsl" href="' . esc_url(home_url('/sitemap.xsl')) . '"?>';
        }
        
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"';
        $xml .= ' xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"';
        $xml .= ' xsi:schemaLocation="http://www.sitemaps.org/schemas/sitemap/0.9 http://www.sitemaps.org/schemas/sitemap/0.9/sitemap.xsd"';
        $xml .= '>';
        
        // Get authors with published posts
        $authors = $this->get_authors_for_sitemap();
        
        foreach ($authors as $author) {
            $xml .= $this->get_url_entry_for_author($author);
        }
        
        $xml .= '</urlset>';
        
        return $xml;
    }

    /**
     * Check if the site is multilingual.
     *
     * @since    1.0.0
     * @return   boolean    True if the site is multilingual, false otherwise.
     */
    private function is_multilingual_site() {
        // Check for WPML
        if (defined('ICL_SITEPRESS_VERSION')) {
            return true;
        }
        
        // Check for Polylang
        if (function_exists('pll_languages_list')) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Get the enabled post types for the sitemap.
     *
     * @since    1.0.0
     * @return   array    The enabled post types.
     */
    private function get_enabled_post_types() {
        $enabled_post_types = array();
        
        if (isset($this->options['sitemap_items']) && is_array($this->options['sitemap_items'])) {
            foreach ($this->options['sitemap_items'] as $post_type => $enabled) {
                if ($enabled && post_type_exists($post_type)) {
                    $enabled_post_types[] = $post_type;
                }
            }
        }
        
        return $enabled_post_types;
    }
    
    /**
     * Get the enabled taxonomies for the sitemap.
     *
     * @since    1.0.0
     * @return   array    The enabled taxonomies.
     */
    private function get_enabled_taxonomies() {
        $enabled_taxonomies = array();
        
        if (isset($this->options['sitemap_taxonomies']) && is_array($this->options['sitemap_taxonomies'])) {
            foreach ($this->options['sitemap_taxonomies'] as $taxonomy => $enabled) {
                if ($enabled && taxonomy_exists($taxonomy)) {
                    $enabled_taxonomies[] = $taxonomy;
                }
            }
        }
        
        return $enabled_taxonomies;
    }
    
    /**
     * Check if there are authors with published posts.
     *
     * @since    1.0.0
     * @return   boolean    True if there are authors with published posts, false otherwise.
     */
    private function has_authors_with_posts() {
        $authors = get_users(array(
            'who' => 'authors',
            'has_published_posts' => true,
        ));
        
        return !empty($authors);
    }
    
    /**
     * Get posts for the sitemap.
     *
     * @since    1.0.0
     * @param    string    $post_type    The post type to get posts for.
     * @return   array                   The posts.
     */
    private function get_posts_for_sitemap($post_type) {
        $excluded_posts = isset($this->options['sitemap_exclude_posts']) ? $this->options['sitemap_exclude_posts'] : array();
        
        $args = array(
            'post_type' => $post_type,
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'post__not_in' => $excluded_posts,
            'orderby' => 'modified',
            'order' => 'DESC',
        );
        
        $query = new WP_Query($args);
        
        return $query->posts;
    }
    
    /**
     * Get terms for the sitemap.
     *
     * @since    1.0.0
     * @param    string    $taxonomy    The taxonomy to get terms for.
     * @return   array                  The terms.
     */
    private function get_terms_for_sitemap($taxonomy) {
        $excluded_terms = isset($this->options['sitemap_exclude_terms']) ? $this->options['sitemap_exclude_terms'] : array();
        
        $args = array(
            'taxonomy' => $taxonomy,
            'hide_empty' => true,
            'exclude' => $excluded_terms,
        );
        
        $terms = get_terms($args);
        
        return $terms;
    }
    
    /**
     * Get authors for the sitemap.
     *
     * @since    1.0.0
     * @return   array    The authors.
     */
    private function get_authors_for_sitemap() {
        $authors = get_users(array(
            'who' => 'authors',
            'has_published_posts' => true,
        ));
        
        return $authors;
    }
    
    /**
     * Generate the Google News sitemap.
     *
     * @since    1.0.0
     * @return   string    The XML content of the Google News sitemap.
     */
    public function generate_news_sitemap() {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:news="http://www.google.com/schemas/sitemap-news/0.9">';

        // Get the publication name from options
        $publication_name = isset($this->options['sitemap_news_name']) ? $this->options['sitemap_news_name'] : get_bloginfo('name');
        
        // Get post types for Google News
        $news_post_types = array();
        if (isset($this->options['sitemap_news_post_types']) && is_array($this->options['sitemap_news_post_types'])) {
            foreach ($this->options['sitemap_news_post_types'] as $post_type => $enabled) {
                if ($enabled) {
                    $news_post_types[] = $post_type;
                }
            }
        } else {
            // Default to regular posts if no specific types are set
            $news_post_types = array('post');
        }
        
        // Get news articles published in the last 48 hours
        $args = array(
            'post_type' => $news_post_types,
            'post_status' => 'publish',
            'posts_per_page' => 1000, // Google News limit
            'date_query' => array(
                array(
                    'after' => '2 days ago',
                ),
            ),
        );
        
        $query = new WP_Query($args);
        
        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                
                $post_id = get_the_ID();
                $post_url = get_permalink();
                $post_title = get_the_title();
                $post_date = get_the_date('Y-m-d\\TH:i:sP');
                
                // Get post categories
                $categories = get_the_category();
                $category_name = !empty($categories) ? $categories[0]->name : 'News';
                
                // Add entry to sitemap
                $xml .= '<url>';
                $xml .= '<loc>' . esc_url($post_url) . '</loc>';
                $xml .= '<news:news>';
                $xml .= '<news:publication>';
                $xml .= '<news:name>' . esc_html($publication_name) . '</news:name>';
                $xml .= '<news:language>' . esc_html(substr(get_locale(), 0, 2)) . '</news:language>';
                $xml .= '</news:publication>';
                $xml .= '<news:publication_date>' . esc_html($post_date) . '</news:publication_date>';
                $xml .= '<news:title>' . esc_html($post_title) . '</news:title>';
                
                // Optional: Add genres (categories)
                $xml .= '<news:genres>' . esc_html($category_name) . '</news:genres>';
                
                $xml .= '</news:news>';
                $xml .= '</url>';
            }
        }
        
        wp_reset_postdata();
        
        $xml .= '</urlset>';
        return $xml;
    }
}
