<?php
/**
 * Fired during plugin activation
 *
 * @link       https://plugins.quickpr.co/
 * @since      1.0.0
 *
 * @package    Quick_Sitemap
 * @subpackage Quick_Sitemap/includes
 */

/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since      1.0.0
 * @package    Quick_Sitemap
 * @subpackage Quick_Sitemap/includes
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Sitemap_Activator {

    /**
     * Activate the plugin.
     *
     * Set up default settings and flush rewrite rules.
     *
     * @since    1.0.0
     */
    public static function activate() {
        // Set default options
        self::set_default_options();
        
        // Flush rewrite rules to ensure sitemap URLs work
        flush_rewrite_rules();
    }
    
    /**
     * Set default options for the plugin.
     *
     * @since    1.0.0
     */
    private static function set_default_options() {
        $default_options = array(
            'sitemap_enabled' => true,
            'sitemap_items' => array(
                'post' => true,
                'page' => true,
                'attachment' => false,
            ),
            'sitemap_taxonomies' => array(
                'category' => true,
                'post_tag' => true,
            ),
            'sitemap_include_images' => true,
            'sitemap_include_videos' => true,
            'sitemap_google_news' => false,
            'sitemap_exclude_posts' => array(),
            'sitemap_exclude_terms' => array(),
            'sitemap_priority' => array(
                'home' => '1.0',
                'post' => '0.7',
                'page' => '0.8',
                'category' => '0.5',
                'post_tag' => '0.5',
                'author' => '0.3',
            ),
            'sitemap_frequency' => array(
                'home' => 'daily',
                'post' => 'weekly',
                'page' => 'monthly',
                'category' => 'weekly',
                'post_tag' => 'weekly',
                'author' => 'monthly',
            ),
            'sitemap_ping_search_engines' => true,
            'sitemap_ping_services' => array(
                'google' => true,
                'bing' => true,
            ),
            'sitemap_stylesheet' => true,
        );
        
        // Get custom post types
        $custom_post_types = get_post_types(array(
            'public' => true,
            '_builtin' => false,
        ), 'objects');
        
        // Add custom post types to default options
        foreach ($custom_post_types as $post_type) {
            $default_options['sitemap_items'][$post_type->name] = true;
            $default_options['sitemap_priority'][$post_type->name] = '0.5';
            $default_options['sitemap_frequency'][$post_type->name] = 'weekly';
        }
        
        // Get custom taxonomies
        $custom_taxonomies = get_taxonomies(array(
            'public' => true,
            '_builtin' => false,
        ), 'objects');
        
        // Add custom taxonomies to default options
        foreach ($custom_taxonomies as $taxonomy) {
            $default_options['sitemap_taxonomies'][$taxonomy->name] = true;
            $default_options['sitemap_priority'][$taxonomy->name] = '0.5';
            $default_options['sitemap_frequency'][$taxonomy->name] = 'weekly';
        }
        
        // Add default options to database if they don't exist
        if (!get_option('quick_sitemap_options')) {
            add_option('quick_sitemap_options', $default_options);
        }
    }
}
