<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://plugins.quickpr.co/
 * @since      1.0.0
 *
 * @package    Quick_Sitemap
 * @subpackage Quick_Sitemap/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Quick_Sitemap
 * @subpackage Quick_Sitemap/admin
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Sitemap_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * The options of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $options    The options of this plugin.
     */
    private $options;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->options = get_option('quick_sitemap_options', array());
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/quick-sitemap-admin.css', array(), $this->version, 'all');
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/quick-sitemap-admin.js', array('jquery'), $this->version, false);
        
        wp_localize_script($this->plugin_name, 'quick_sitemap_params', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('quick-sitemap-nonce'),
            'i18n' => array(
                'regenerating' => __('Regenerating sitemap...', 'quick-sitemap'),
                'regenerate_success' => __('Sitemap regenerated successfully!', 'quick-sitemap'),
                'regenerate_error' => __('Error regenerating sitemap:', 'quick-sitemap'),
                'pinging' => __('Pinging search engines...', 'quick-sitemap'),
                'ping_success' => __('Search engines pinged successfully!', 'quick-sitemap'),
                'ping_error' => __('Error pinging search engines:', 'quick-sitemap'),
            ),
        ));
    }

    /**
     * Add menu items to the admin menu.
     *
     * @since    1.0.0
     */
    public function add_plugin_admin_menu() {
        // Main menu item
        add_menu_page(
            __('Quick Sitemap', 'quick-sitemap'),
            __('🗺️ Sitemap', 'quick-sitemap'),
            'manage_options',
            $this->plugin_name,
            array($this, 'display_plugin_admin_page'),
            'dashicons-networking',
            81
        );
        
        // Settings submenu
        add_submenu_page(
            $this->plugin_name,
            __('Settings', 'quick-sitemap'),
            __('Settings', 'quick-sitemap'),
            'manage_options',
            $this->plugin_name,
            array($this, 'display_plugin_admin_page')
        );
        
        // View Sitemap submenu
        add_submenu_page(
            $this->plugin_name,
            __('View Sitemap', 'quick-sitemap'),
            __('View Sitemap', 'quick-sitemap'),
            'manage_options',
            $this->plugin_name . '-view',
            array($this, 'display_plugin_view_page')
        );
    }

    /**
     * Add action links to the plugin list table.
     *
     * @since    1.0.0
     * @param    array    $links    An array of plugin action links.
     * @return   array              An array of plugin action links.
     */
    public function add_action_links($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=' . $this->plugin_name) . '">' . __('Settings', 'quick-sitemap') . '</a>';
        $view_link = '<a href="' . home_url('/sitemap.xml') . '" target="_blank">' . __('View Sitemap', 'quick-sitemap') . '</a>';
        
        array_unshift($links, $settings_link, $view_link);
        
        return $links;
    }

    /**
     * Register plugin settings.
     *
     * @since    1.0.0
     */
    public function register_settings() {
        register_setting(
            'quick_sitemap_options',
            'quick_sitemap_options',
            array($this, 'validate_options')
        );
    }

    /**
     * Validate plugin options.
     *
     * @since    1.0.0
     * @param    array    $input    The options to validate.
     * @return   array              The validated options.
     */
    public function validate_options($input) {
        $valid = array();
        
        // Sitemap enabled
        $valid['sitemap_enabled'] = isset($input['sitemap_enabled']) ? (bool) $input['sitemap_enabled'] : true;
        
        // Post types
        $valid['sitemap_items'] = array();
        $post_types = get_post_types(array('public' => true), 'objects');
        foreach ($post_types as $post_type) {
            $valid['sitemap_items'][$post_type->name] = isset($input['sitemap_items'][$post_type->name]) ? (bool) $input['sitemap_items'][$post_type->name] : false;
        }
        
        // Taxonomies
        $valid['sitemap_taxonomies'] = array();
        $taxonomies = get_taxonomies(array('public' => true), 'objects');
        foreach ($taxonomies as $taxonomy) {
            $valid['sitemap_taxonomies'][$taxonomy->name] = isset($input['sitemap_taxonomies'][$taxonomy->name]) ? (bool) $input['sitemap_taxonomies'][$taxonomy->name] : false;
        }
        
        // Include images
        $valid['sitemap_include_images'] = isset($input['sitemap_include_images']) ? (bool) $input['sitemap_include_images'] : true;
        
        // Include videos
        $valid['sitemap_include_videos'] = isset($input['sitemap_include_videos']) ? (bool) $input['sitemap_include_videos'] : true;
        
        // Google News
        $valid['sitemap_google_news'] = isset($input['sitemap_google_news']) ? (bool) $input['sitemap_google_news'] : false;
        
        // Excluded posts
        $valid['sitemap_exclude_posts'] = isset($input['sitemap_exclude_posts']) && is_array($input['sitemap_exclude_posts']) ? array_map('intval', $input['sitemap_exclude_posts']) : array();
        
        // Excluded terms
        $valid['sitemap_exclude_terms'] = isset($input['sitemap_exclude_terms']) && is_array($input['sitemap_exclude_terms']) ? array_map('intval', $input['sitemap_exclude_terms']) : array();
        
        // Priorities
        $valid['sitemap_priority'] = array();
        $priority_items = array('home', 'post', 'page', 'category', 'post_tag', 'author');
        $custom_post_types = get_post_types(array('public' => true, '_builtin' => false), 'names');
        $priority_items = array_merge($priority_items, $custom_post_types);
        $custom_taxonomies = get_taxonomies(array('public' => true, '_builtin' => false), 'names');
        $priority_items = array_merge($priority_items, $custom_taxonomies);
        
        foreach ($priority_items as $item) {
            $valid['sitemap_priority'][$item] = isset($input['sitemap_priority'][$item]) ? $this->validate_priority($input['sitemap_priority'][$item]) : '0.5';
        }
        
        // Frequencies
        $valid['sitemap_frequency'] = array();
        $frequency_items = $priority_items; // Same items as priorities
        $valid_frequencies = array('always', 'hourly', 'daily', 'weekly', 'monthly', 'yearly', 'never');
        
        foreach ($frequency_items as $item) {
            $valid['sitemap_frequency'][$item] = isset($input['sitemap_frequency'][$item]) && in_array($input['sitemap_frequency'][$item], $valid_frequencies) ? $input['sitemap_frequency'][$item] : 'weekly';
        }
        
        // Ping search engines
        $valid['sitemap_ping_search_engines'] = isset($input['sitemap_ping_search_engines']) ? (bool) $input['sitemap_ping_search_engines'] : true;
        
        // Ping services
        $valid['sitemap_ping_services'] = array();
        $services = array('google', 'bing');
        foreach ($services as $service) {
            $valid['sitemap_ping_services'][$service] = isset($input['sitemap_ping_services'][$service]) ? (bool) $input['sitemap_ping_services'][$service] : true;
        }
        
        // Stylesheet
        $valid['sitemap_stylesheet'] = isset($input['sitemap_stylesheet']) ? (bool) $input['sitemap_stylesheet'] : true;
        
        return $valid;
    }

    /**
     * Validate priority value.
     *
     * @since    1.0.0
     * @param    string    $priority    The priority to validate.
     * @return   string                 The validated priority.
     */
    private function validate_priority($priority) {
        $priority = (float) $priority;
        
        if ($priority < 0) {
            $priority = 0;
        } elseif ($priority > 1) {
            $priority = 1;
        }
        
        return number_format($priority, 1);
    }

    /**
     * Display the plugin admin page.
     *
     * @since    1.0.0
     */
    public function display_plugin_admin_page() {
        include_once 'partials/quick-sitemap-admin-display.php';
    }

    /**
     * Display the plugin view page.
     *
     * @since    1.0.0
     */
    public function display_plugin_view_page() {
        include_once 'partials/quick-sitemap-admin-view.php';
    }

    /**
     * AJAX handler for pinging search engines.
     *
     * @since    1.0.0
     */
    public function ajax_ping_search_engines() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'quick-sitemap-nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'quick-sitemap')));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'quick-sitemap')));
        }
        
        // Ping search engines
        $sitemap_url = home_url('/sitemap.xml');
        $results = array();
        
        // Ping Google
        if (isset($this->options['sitemap_ping_services']['google']) && $this->options['sitemap_ping_services']['google']) {
            $google_result = $this->ping_google($sitemap_url);
            $results['google'] = $google_result;
        }
        
        // Ping Bing
        if (isset($this->options['sitemap_ping_services']['bing']) && $this->options['sitemap_ping_services']['bing']) {
            $bing_result = $this->ping_bing($sitemap_url);
            $results['bing'] = $bing_result;
        }
        
        wp_send_json_success(array(
            'message' => __('Search engines pinged successfully.', 'quick-sitemap'),
            'results' => $results,
        ));
    }

    /**
     * AJAX handler for regenerating the sitemap.
     *
     * @since    1.0.0
     */
    public function ajax_regenerate_sitemap() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'quick-sitemap-nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'quick-sitemap')));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'quick-sitemap')));
        }
        
        // Regenerate sitemap
        $generator = new Quick_Sitemap_Generator();
        $sitemap = $generator->generate_sitemap_index();
        
        // Store the last regeneration time
        update_option('quick_sitemap_last_generated', time());
        
        wp_send_json_success(array(
            'message' => __('Sitemap regenerated successfully.', 'quick-sitemap'),
            'time' => date_i18n(get_option('date_format') . ' ' . get_option('time_format'), time()),
        ));
    }

    /**
     * Ping Google about the sitemap.
     *
     * @since    1.0.0
     * @param    string    $sitemap_url    The URL of the sitemap.
     * @return   array                     The result of the ping.
     */
    private function ping_google($sitemap_url) {
        $ping_url = 'https://www.google.com/ping?sitemap=' . urlencode($sitemap_url);
        
        $response = wp_remote_get($ping_url);
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message(),
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code === 200) {
            return array(
                'success' => true,
                'message' => __('Google successfully pinged.', 'quick-sitemap'),
            );
        } else {
            return array(
                'success' => false,
                'message' => sprintf(__('Google ping failed with response code: %s', 'quick-sitemap'), $response_code),
            );
        }
    }

    /**
     * Ping Bing about the sitemap.
     *
     * @since    1.0.0
     * @param    string    $sitemap_url    The URL of the sitemap.
     * @return   array                     The result of the ping.
     */
    private function ping_bing($sitemap_url) {
        $ping_url = 'https://www.bing.com/ping?sitemap=' . urlencode($sitemap_url);
        
        $response = wp_remote_get($ping_url);
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message(),
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code === 200) {
            return array(
                'success' => true,
                'message' => __('Bing successfully pinged.', 'quick-sitemap'),
            );
        } else {
            return array(
                'success' => false,
                'message' => sprintf(__('Bing ping failed with response code: %s', 'quick-sitemap'), $response_code),
            );
        }
    }
}
