/**
 * Public-facing JavaScript for Quick Push Notifications
 *
 * Handles the subscription process and notification display
 */
(function($) {
    'use strict';

    // Firebase Messaging instance
    let messaging = null;

    // DOM elements
    const subscriptionButton = $('#qpn-subscription-button');
    const subscribeText = $('#qpn-subscribe-text');
    const unsubscribeText = $('#qpn-unsubscribe-text');
    const dialog = $('#qpn-subscription-dialog');
    const dialogTitle = $('#qpn-dialog-title');
    const dialogText = $('#qpn-dialog-text');
    const dialogConfirm = $('#qpn-dialog-confirm');
    const dialogCancel = $('#qpn-dialog-cancel');
    const permissionDenied = $('#qpn-permission-denied');

    // Initialize Firebase
    function initializeFirebase() {
        try {
            // Initialize Firebase with the provided config
            if (!firebase.apps.length) {
                firebase.initializeApp(qpn_vars.firebase_config);
            }

            // Get messaging instance
            messaging = firebase.messaging();

            // Set background message handler
            messaging.onMessage((payload) => {
                console.log('Message received:', payload);
                
                // Display the notification if the browser doesn't handle it automatically
                if ('Notification' in window && Notification.permission === 'granted') {
                    const notificationTitle = payload.data.notification.title;
                    const notificationOptions = {
                        body: payload.data.notification.body,
                        icon: payload.data.notification.icon,
                        image: payload.data.notification.image,
                        data: payload.data.notification.data,
                        actions: payload.data.notification.actions
                    };

                    const notification = new Notification(notificationTitle, notificationOptions);
                    
                    notification.onclick = function() {
                        const url = payload.data.notification.click_action || '/';
                        window.open(url, '_blank');
                        notification.close();
                    };
                }
            });

            // Check if already subscribed
            checkSubscriptionStatus();

            // Show subscription button
            $('#qpn-subscription-button-container').show();
        } catch (error) {
            console.error('Error initializing Firebase:', error);
        }
    }

    // Check if the user is already subscribed
    function checkSubscriptionStatus() {
        // Check if service worker is available
        if ('serviceWorker' in navigator) {
            navigator.serviceWorker.register(qpn_vars.service_worker_url)
                .then((registration) => {
                    console.log('Service Worker registered with scope:', registration.scope);
                    
                    // Set messaging service worker
                    messaging.useServiceWorker(registration);
                    
                    // Check current permission
                    checkNotificationPermission();
                    
                    // Get token if permission is granted
                    if (Notification.permission === 'granted') {
                        getToken();
                    }
                })
                .catch((error) => {
                    console.error('Service Worker registration failed:', error);
                });
        } else {
            console.warn('Service workers are not supported in this browser');
            subscriptionButton.prop('disabled', true);
            subscriptionButton.text('Notifications not supported');
        }
    }

    // Check notification permission
    function checkNotificationPermission() {
        if ('Notification' in window) {
            if (Notification.permission === 'granted') {
                // Already subscribed
                subscribeText.hide();
                unsubscribeText.show();
            } else if (Notification.permission === 'denied') {
                // Permission denied
                subscribeText.show();
                unsubscribeText.hide();
            } else {
                // Permission not asked yet
                subscribeText.show();
                unsubscribeText.hide();
            }
        } else {
            console.warn('Notifications are not supported in this browser');
            subscriptionButton.prop('disabled', true);
            subscriptionButton.text('Notifications not supported');
        }
    }

    // Get Firebase messaging token
    function getToken() {
        messaging.getToken()
            .then((currentToken) => {
                if (currentToken) {
                    console.log('Token retrieved');
                    saveSubscription(currentToken);
                } else {
                    console.log('No token available');
                }
            })
            .catch((error) => {
                console.error('Error retrieving token:', error);
            });
    }

    // Save subscription to the server
    function saveSubscription(token) {
        const subscription = {
            endpoint: token,
            keys: {
                auth: 'auth-secret',
                p256dh: 'p256dh-key'
            }
        };

        // Send subscription to server
        $.ajax({
            url: qpn_vars.rest_url + 'subscribe',
            method: 'POST',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', qpn_vars.nonce);
            },
            data: {
                subscription: subscription
            },
            success: function(response) {
                console.log('Subscription saved:', response);
            },
            error: function(error) {
                console.error('Error saving subscription:', error);
            }
        });
    }

    // Delete subscription from the server
    function deleteSubscription(token) {
        $.ajax({
            url: qpn_vars.rest_url + 'unsubscribe',
            method: 'POST',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', qpn_vars.nonce);
            },
            data: {
                endpoint: token
            },
            success: function(response) {
                console.log('Subscription deleted:', response);
            },
            error: function(error) {
                console.error('Error deleting subscription:', error);
            }
        });
    }

    // Subscribe to push notifications
    function subscribeToPushNotifications() {
        Notification.requestPermission()
            .then((permission) => {
                if (permission === 'granted') {
                    // Get token and save subscription
                    getToken();
                    
                    // Update button state
                    subscribeText.hide();
                    unsubscribeText.show();
                    
                    // Close dialog
                    dialog.hide();
                } else if (permission === 'denied') {
                    // Show permission denied message
                    dialogTitle.text(qpn_vars.notification_permission_title);
                    dialogText.hide();
                    permissionDenied.show();
                    dialogConfirm.hide();
                }
            })
            .catch((error) => {
                console.error('Error requesting permission:', error);
            });
    }

    // Unsubscribe from push notifications
    function unsubscribeFromPushNotifications() {
        messaging.getToken()
            .then((currentToken) => {
                if (currentToken) {
                    // Delete token
                    messaging.deleteToken(currentToken)
                        .then(() => {
                            console.log('Token deleted');
                            
                            // Delete subscription from server
                            deleteSubscription(currentToken);
                            
                            // Update button state
                            subscribeText.show();
                            unsubscribeText.hide();
                        })
                        .catch((error) => {
                            console.error('Error deleting token:', error);
                        });
                }
            })
            .catch((error) => {
                console.error('Error retrieving token:', error);
            });
    }

    // Event listeners
    $(document).ready(function() {
        // Initialize Firebase when document is ready
        initializeFirebase();

        // Subscription button click
        subscriptionButton.on('click', function() {
            if (Notification.permission === 'granted') {
                // Already subscribed, unsubscribe
                if (confirm(qpn_vars.notification_permission_denied_text)) {
                    unsubscribeFromPushNotifications();
                }
            } else if (Notification.permission === 'denied') {
                // Permission denied, show instructions
                dialogTitle.text(qpn_vars.notification_permission_title);
                dialogText.hide();
                permissionDenied.show();
                dialogConfirm.hide();
                dialog.show();
            } else {
                // Ask for permission
                dialogTitle.text(qpn_vars.subscribe_dialog_title);
                dialogText.show();
                permissionDenied.hide();
                dialogConfirm.show();
                dialog.show();
            }
        });

        // Dialog confirm button
        dialogConfirm.on('click', function() {
            subscribeToPushNotifications();
        });

        // Dialog cancel button
        dialogCancel.on('click', function() {
            dialog.hide();
        });

        // Dialog close button
        $('.qpn-dialog-close').on('click', function() {
            dialog.hide();
        });

        // Close dialog when clicking outside
        $(window).on('click', function(event) {
            if ($(event.target).is(dialog)) {
                dialog.hide();
            }
        });
    });

    // Add shortcode handler
    window.qpnInitializeShortcode = function() {
        // For dynamically loaded content
        checkSubscriptionStatus();
    };

})(jQuery);
