<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://plugins.quickpr.co/
 * @since      1.0.0
 *
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for
 * the public-facing side of the site.
 *
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/public
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Push_Notifications_Public {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/quick-push-notifications-public.css', array(), $this->version, 'all');
    }

    /**
     * Register the JavaScript for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        // Firebase scripts
        wp_enqueue_script('firebase-app', 'https://www.gstatic.com/firebasejs/8.10.0/firebase-app.js', array(), '8.10.0', true);
        wp_enqueue_script('firebase-messaging', 'https://www.gstatic.com/firebasejs/8.10.0/firebase-messaging.js', array('firebase-app'), '8.10.0', true);
        
        // Plugin script
        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/quick-push-notifications-public.js', array('jquery', 'firebase-app', 'firebase-messaging'), $this->version, true);
        
        // Localize script
        wp_localize_script($this->plugin_name, 'qpn_vars', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'rest_url' => rest_url('quick-push-notifications/v1/'),
            'nonce' => wp_create_nonce('wp_rest'),
            'firebase_config' => Quick_Push_Notifications_Firebase::get_firebase_config(),
            'service_worker_url' => home_url('/firebase-messaging-sw.js'),
            'manifest_url' => rest_url('quick-push-notifications/v1/manifest.json'),
            'subscribe_button_text' => __('Subscribe to Notifications', 'quick-push-notifications'),
            'unsubscribe_button_text' => __('Unsubscribe from Notifications', 'quick-push-notifications'),
            'subscribe_dialog_title' => __('Subscribe to Notifications', 'quick-push-notifications'),
            'subscribe_dialog_text' => __('Would you like to receive notifications from this website?', 'quick-push-notifications'),
            'subscribe_dialog_confirm' => __('Yes, Subscribe', 'quick-push-notifications'),
            'subscribe_dialog_cancel' => __('No, Thanks', 'quick-push-notifications'),
            'notification_permission_title' => __('Notification Permission', 'quick-push-notifications'),
            'notification_permission_text' => __('Please allow notifications in your browser settings to receive notifications from this website.', 'quick-push-notifications'),
            'notification_permission_denied_text' => __('You have denied notification permission. Please enable notifications in your browser settings to receive notifications from this website.', 'quick-push-notifications')
        ));
        
        // Generate service worker file
        $this->generate_service_worker();
    }

    /**
     * Add manifest link to the head.
     *
     * @since    1.0.0
     */
    public function add_manifest_link() {
        echo '<link rel="manifest" href="' . esc_url(rest_url('quick-push-notifications/v1/manifest.json')) . '">';
    }

    /**
     * Add subscription dialog to the footer.
     *
     * @since    1.0.0
     */
    public function add_subscription_dialog() {
        include_once 'partials/quick-push-notifications-public-dialog.php';
    }

    /**
     * Generate the service worker file.
     *
     * @since    1.0.0
     */
    private function generate_service_worker() {
        $file_path = ABSPATH . 'firebase-messaging-sw.js';
        
        // Get Firebase configuration
        $firebase_config = Quick_Push_Notifications_Firebase::get_firebase_config();
        
        // Create service worker content
        $content = "// Firebase Messaging Service Worker\n\n";
        $content .= "importScripts('https://www.gstatic.com/firebasejs/8.10.0/firebase-app.js');\n";
        $content .= "importScripts('https://www.gstatic.com/firebasejs/8.10.0/firebase-messaging.js');\n\n";
        
        $content .= "// Firebase configuration\n";
        $content .= "const firebaseConfig = {\n";
        $content .= "  apiKey: '" . esc_js($firebase_config['apiKey']) . "',\n";
        $content .= "  authDomain: '" . esc_js($firebase_config['authDomain']) . "',\n";
        $content .= "  projectId: '" . esc_js($firebase_config['projectId']) . "',\n";
        $content .= "  storageBucket: '" . esc_js($firebase_config['storageBucket']) . "',\n";
        $content .= "  messagingSenderId: '" . esc_js($firebase_config['messagingSenderId']) . "',\n";
        $content .= "  appId: '" . esc_js($firebase_config['appId']) . "'\n";
        $content .= "};\n\n";
        
        $content .= "// Initialize Firebase\n";
        $content .= "firebase.initializeApp(firebaseConfig);\n";
        $content .= "const messaging = firebase.messaging();\n\n";
        
        $content .= "// Background message handler\n";
        $content .= "messaging.setBackgroundMessageHandler(function(payload) {\n";
        $content .= "  console.log('[firebase-messaging-sw.js] Received background message ', payload);\n";
        $content .= "  const notificationTitle = payload.data.notification.title;\n";
        $content .= "  const notificationOptions = {\n";
        $content .= "    body: payload.data.notification.body,\n";
        $content .= "    icon: payload.data.notification.icon,\n";
        $content .= "    image: payload.data.notification.image,\n";
        $content .= "    data: payload.data.notification.data,\n";
        $content .= "    actions: payload.data.notification.actions\n";
        $content .= "  };\n\n";
        $content .= "  return self.registration.showNotification(notificationTitle, notificationOptions);\n";
        $content .= "});\n\n";
        
        $content .= "// Notification click handler\n";
        $content .= "self.addEventListener('notificationclick', function(event) {\n";
        $content .= "  console.log('[firebase-messaging-sw.js] Notification click ', event);\n";
        $content .= "  event.notification.close();\n\n";
        $content .= "  // Handle notification click\n";
        $content .= "  const clickAction = event.notification.data && event.notification.data.url ? event.notification.data.url : '/';\n\n";
        $content .= "  // Check if a specific action was clicked\n";
        $content .= "  if (event.action) {\n";
        $content .= "    // Handle custom actions here\n";
        $content .= "    console.log('Action clicked:', event.action);\n";
        $content .= "  }\n\n";
        $content .= "  event.waitUntil(\n";
        $content .= "    clients.matchAll({type: 'window'}).then(function(windowClients) {\n";
        $content .= "      // Check if there is already a window/tab open with the target URL\n";
        $content .= "      for (var i = 0; i < windowClients.length; i++) {\n";
        $content .= "        var client = windowClients[i];\n";
        $content .= "        // If so, focus on it\n";
        $content .= "        if (client.url === clickAction && 'focus' in client) {\n";
        $content .= "          return client.focus();\n";
        $content .= "        }\n";
        $content .= "      }\n";
        $content .= "      // If not, open a new window/tab\n";
        $content .= "      if (clients.openWindow) {\n";
        $content .= "        return clients.openWindow(clickAction);\n";
        $content .= "      }\n";
        $content .= "    })\n";
        $content .= "  );\n";
        $content .= "});\n";
        
        // Write to file
        file_put_contents($file_path, $content);
    }
}
