<?php
/**
 * Handles Firebase integration
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 */

/**
 * Handles Firebase integration.
 *
 * This class handles the integration with Firebase Cloud Messaging for sending push notifications.
 *
 * @since      1.0.0
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 * @author     Quick Plugins <info@example.com>
 */
class Quick_Push_Notifications_Firebase {

    /**
     * Send a notification to specific subscribers.
     *
     * @since    1.0.0
     * @param    array    $subscriber_ids    Array of subscriber IDs to send the notification to.
     * @param    array    $notification      The notification data.
     * @return   array                       Array of results.
     */
    public static function send_notification_to_subscribers($subscriber_ids, $notification) {
        global $wpdb;
        
        if (empty($subscriber_ids) || !is_array($subscriber_ids)) {
            return array('error' => 'No subscribers specified');
        }
        
        if (empty($notification) || !isset($notification['title']) || !isset($notification['message'])) {
            return array('error' => 'Notification data is incomplete');
        }
        
        $table_name = $wpdb->prefix . 'qpn_subscribers';
        
        // Get subscribers
        $placeholders = implode(',', array_fill(0, count($subscriber_ids), '%d'));
        $query = $wpdb->prepare(
            "SELECT * FROM $table_name WHERE id IN ($placeholders)",
            $subscriber_ids
        );
        
        $subscribers = $wpdb->get_results($query);
        
        if (empty($subscribers)) {
            return array('error' => 'No valid subscribers found');
        }
        
        $results = array(
            'success' => 0,
            'failed' => 0,
            'details' => array()
        );
        
        $notification_log_table = $wpdb->prefix . 'qpn_notification_logs';
        
        foreach ($subscribers as $subscriber) {
            $subscription_data = json_decode($subscriber->subscription_data, true);
            
            if (empty($subscription_data) || !isset($subscription_data['endpoint'])) {
                $results['failed']++;
                $results['details'][] = array(
                    'subscriber_id' => $subscriber->id,
                    'status' => 'failed',
                    'message' => 'Invalid subscription data'
                );
                continue;
            }
            
            // Prepare notification payload
            $payload = array(
                'notification' => array(
                    'title' => $notification['title'],
                    'body' => $notification['message'],
                    'icon' => isset($notification['icon']) ? $notification['icon'] : '',
                    'click_action' => isset($notification['url']) ? $notification['url'] : home_url(),
                    'data' => array(
                        'url' => isset($notification['url']) ? $notification['url'] : home_url()
                    )
                )
            );
            
            // Add image if provided
            if (!empty($notification['image'])) {
                $payload['notification']['image'] = $notification['image'];
            }
            
            // Add actions if provided
            if (!empty($notification['actions']) && is_array($notification['actions'])) {
                $payload['notification']['actions'] = $notification['actions'];
            }
            
            // Prepare FCM message
            $fields = array(
                'to' => $subscription_data['endpoint'],
                'data' => $payload
            );
            
            // Send the notification
            $response = self::send_push_notification($fields);
            $response_data = json_decode($response, true);
            
            // Log the notification
            $wpdb->insert(
                $notification_log_table,
                array(
                    'subscriber_id' => $subscriber->id,
                    'notification_id' => isset($notification['id']) ? $notification['id'] : 0,
                    'title' => $notification['title'],
                    'message' => $notification['message'],
                    'url' => isset($notification['url']) ? $notification['url'] : '',
                    'icon' => isset($notification['icon']) ? $notification['icon'] : '',
                    'image' => isset($notification['image']) ? $notification['image'] : '',
                    'status' => isset($response_data['success']) && $response_data['success'] > 0 ? 'sent' : 'failed',
                    'response' => $response,
                    'created_at' => current_time('mysql')
                ),
                array('%d', '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s')
            );
            
            if (isset($response_data['success']) && $response_data['success'] > 0) {
                $results['success']++;
                $results['details'][] = array(
                    'subscriber_id' => $subscriber->id,
                    'status' => 'sent',
                    'response' => $response_data
                );
                
                // Update subscriber's last activity
                $wpdb->update(
                    $table_name,
                    array('last_activity' => current_time('mysql')),
                    array('id' => $subscriber->id),
                    array('%s'),
                    array('%d')
                );
            } else {
                $results['failed']++;
                $results['details'][] = array(
                    'subscriber_id' => $subscriber->id,
                    'status' => 'failed',
                    'response' => $response_data
                );
                
                // Check if the subscription is no longer valid
                if (isset($response_data['failure']) && $response_data['failure'] > 0) {
                    $error = isset($response_data['results'][0]['error']) ? $response_data['results'][0]['error'] : '';
                    
                    if ($error === 'NotRegistered' || $error === 'InvalidRegistration') {
                        // Delete the invalid subscriber
                        $wpdb->delete(
                            $table_name,
                            array('id' => $subscriber->id),
                            array('%d')
                        );
                    }
                }
            }
        }
        
        return $results;
    }
    
    /**
     * Send a notification to all subscribers.
     *
     * @since    1.0.0
     * @param    array    $notification    The notification data.
     * @return   array                     Array of results.
     */
    public static function send_notification_to_all($notification) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qpn_subscribers';
        
        // Get all subscriber IDs
        $subscriber_ids = $wpdb->get_col("SELECT id FROM $table_name");
        
        if (empty($subscriber_ids)) {
            return array('error' => 'No subscribers found');
        }
        
        // Send notification to all subscribers
        return self::send_notification_to_subscribers($subscriber_ids, $notification);
    }
    
    /**
     * Send a push notification using Firebase Cloud Messaging.
     *
     * @since    1.0.0
     * @param    array    $fields    The notification fields.
     * @return   string              The response from Firebase.
     */
    private static function send_push_notification($fields) {
        $server_key = get_option('qpn_firebase_server_key', '');
        
        if (empty($server_key)) {
            return json_encode(array('error' => 'Firebase server key is not set'));
        }
        
        // Set POST variables
        $url = 'https://fcm.googleapis.com/fcm/send';
        $headers = array(
            'Authorization: key=' . $server_key,
            'Content-Type: application/json'
        );
        
        // Open connection
        $ch = curl_init();
        
        // Set the url, number of POST vars, POST data
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        // Disabling SSL Certificate support temporarily
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        // Set POST fields
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fields));
        
        // Execute post
        $result = curl_exec($ch);
        
        // Check for errors
        if ($result === false) {
            $result = json_encode(array('error' => curl_error($ch)));
        }
        
        // Close connection
        curl_close($ch);
        
        return $result;
    }
    
    /**
     * Get Firebase configuration for the frontend.
     *
     * @since    1.0.0
     * @return   array    Firebase configuration.
     */
    public static function get_firebase_config() {
        $config = array(
            'apiKey' => get_option('qpn_firebase_api_key', ''),
            'authDomain' => get_option('qpn_firebase_auth_domain', ''),
            'projectId' => get_option('qpn_firebase_project_id', ''),
            'storageBucket' => get_option('qpn_firebase_storage_bucket', ''),
            'messagingSenderId' => get_option('qpn_firebase_messaging_sender_id', ''),
            'appId' => get_option('qpn_firebase_app_id', '')
        );
        
        return $config;
    }
}
