<?php
/**
 * Provide a admin area view for managing notifications
 *
 * This file is used to markup the admin-facing notifications page.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/admin/partials
 */
?>

<div class="wrap">
    <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
    
    <div class="qpn-notifications-container">
        <!-- Actions -->
        <div class="qpn-notifications-actions">
            <a href="<?php echo admin_url('admin.php?page=quick-push-notifications-compose'); ?>" class="button button-primary">
                <span class="dashicons dashicons-edit"></span> <?php _e('Compose New Notification', 'quick-push-notifications'); ?>
            </a>
        </div>
        
        <!-- Filters -->
        <div class="qpn-notifications-filters">
            <div class="qpn-filters-row">
                <div class="qpn-filter">
                    <input type="text" id="qpn_notification_search" placeholder="<?php _e('Search notifications...', 'quick-push-notifications'); ?>">
                </div>
                <div class="qpn-filter">
                    <select id="qpn_status_filter">
                        <option value=""><?php _e('All Statuses', 'quick-push-notifications'); ?></option>
                        <option value="draft"><?php _e('Draft', 'quick-push-notifications'); ?></option>
                        <option value="scheduled"><?php _e('Scheduled', 'quick-push-notifications'); ?></option>
                        <option value="sent"><?php _e('Sent', 'quick-push-notifications'); ?></option>
                        <option value="failed"><?php _e('Failed', 'quick-push-notifications'); ?></option>
                    </select>
                </div>
                <div class="qpn-filter">
                    <button id="qpn_apply_filters" class="button"><?php _e('Apply Filters', 'quick-push-notifications'); ?></button>
                    <button id="qpn_reset_filters" class="button"><?php _e('Reset', 'quick-push-notifications'); ?></button>
                </div>
            </div>
        </div>
        
        <!-- Notifications Table -->
        <div class="qpn-notifications-table-container">
            <table class="wp-list-table widefat fixed striped qpn-notifications-table">
                <thead>
                    <tr>
                        <th class="column-id"><?php _e('ID', 'quick-push-notifications'); ?></th>
                        <th class="column-title"><?php _e('Title', 'quick-push-notifications'); ?></th>
                        <th class="column-message"><?php _e('Message', 'quick-push-notifications'); ?></th>
                        <th class="column-status"><?php _e('Status', 'quick-push-notifications'); ?></th>
                        <th class="column-schedule"><?php _e('Schedule', 'quick-push-notifications'); ?></th>
                        <th class="column-created"><?php _e('Created', 'quick-push-notifications'); ?></th>
                        <th class="column-actions"><?php _e('Actions', 'quick-push-notifications'); ?></th>
                    </tr>
                </thead>
                <tbody id="qpn_notifications_list">
                    <!-- Notification rows will be loaded via AJAX -->
                    <tr class="qpn-loading-row">
                        <td colspan="7"><?php _e('Loading notifications...', 'quick-push-notifications'); ?></td>
                    </tr>
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        <div class="qpn-pagination">
            <div class="qpn-pagination-info">
                <span id="qpn_pagination_info"><?php _e('Loading...', 'quick-push-notifications'); ?></span>
            </div>
            <div class="qpn-pagination-controls">
                <button id="qpn_prev_page" class="button" disabled><?php _e('Previous', 'quick-push-notifications'); ?></button>
                <span id="qpn_current_page">1</span>
                <button id="qpn_next_page" class="button" disabled><?php _e('Next', 'quick-push-notifications'); ?></button>
            </div>
        </div>
    </div>
</div>

<!-- Notification Details Modal -->
<div id="qpn_notification_modal" class="qpn-modal">
    <div class="qpn-modal-content">
        <span class="qpn-modal-close">&times;</span>
        <h2><?php _e('Notification Details', 'quick-push-notifications'); ?></h2>
        <div id="qpn_notification_details">
            <!-- Notification details will be loaded via AJAX -->
        </div>
    </div>
</div>

<style>
/* Notifications Table */
.qpn-notifications-container {
    margin-top: 20px;
}

.qpn-notifications-actions {
    margin-bottom: 20px;
}

.qpn-notifications-filters {
    background: #fff;
    border-radius: 5px;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    margin-bottom: 20px;
    padding: 15px;
}

.qpn-filters-row {
    display: flex;
    flex-wrap: wrap;
    gap: 10px;
    align-items: center;
}

.qpn-filter {
    margin-bottom: 10px;
}

.qpn-notifications-table-container {
    background: #fff;
    border-radius: 5px;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    margin-bottom: 20px;
    overflow-x: auto;
}

.qpn-notifications-table {
    width: 100%;
}

.qpn-notifications-table .column-id {
    width: 50px;
}

.qpn-notifications-table .column-title {
    width: 20%;
}

.qpn-notifications-table .column-message {
    width: 30%;
}

.qpn-notifications-table .column-status {
    width: 80px;
}

.qpn-notifications-table .column-schedule,
.qpn-notifications-table .column-created {
    width: 150px;
}

.qpn-notifications-table .column-actions {
    width: 150px;
    text-align: right;
}

.qpn-loading-row td {
    text-align: center;
    padding: 20px;
}

/* Status Indicators */
.qpn-status {
    background: #f0f0f0;
    border-radius: 3px;
    color: #666;
    display: inline-block;
    font-size: 12px;
    padding: 3px 8px;
}

.qpn-status-draft {
    background: #e5e5e5;
    color: #72777c;
}

.qpn-status-scheduled {
    background: #f8dda7;
    color: #94660c;
}

.qpn-status-sent {
    background: #c6e1c6;
    color: #5b841b;
}

.qpn-status-failed {
    background: #f1adad;
    color: #a00;
}

/* Pagination */
.qpn-pagination {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: #fff;
    border-radius: 5px;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    margin-bottom: 20px;
    padding: 10px 15px;
}

.qpn-pagination-controls {
    display: flex;
    align-items: center;
    gap: 10px;
}

/* Modal */
.qpn-modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    overflow: auto;
    background-color: rgba(0, 0, 0, 0.4);
}

.qpn-modal-content {
    background-color: #fefefe;
    margin: 10% auto;
    padding: 20px;
    border: 1px solid #ddd;
    border-radius: 5px;
    width: 80%;
    max-width: 600px;
    position: relative;
}

.qpn-modal-close {
    color: #aaa;
    float: right;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
}

.qpn-modal-close:hover,
.qpn-modal-close:focus {
    color: black;
    text-decoration: none;
    cursor: pointer;
}

/* Notification Preview */
.qpn-notification-preview {
    background: #fff;
    border-radius: 5px;
    box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
    margin-top: 15px;
    padding: 15px;
}

.qpn-notification-preview-title {
    font-size: 16px;
    font-weight: 600;
    margin: 0 0 5px;
}

.qpn-notification-preview-body {
    font-size: 14px;
    margin: 0 0 10px;
}

.qpn-notification-preview-icon {
    float: left;
    height: 40px;
    margin-right: 10px;
    width: 40px;
}

.qpn-notification-preview-content {
    overflow: hidden;
}

.qpn-notification-preview-image {
    margin-top: 10px;
}

.qpn-notification-preview-image img {
    max-width: 100%;
    height: auto;
}
</style>

<script type="text/javascript">
jQuery(document).ready(function($) {
    // Variables
    var currentPage = 1;
    var itemsPerPage = 20;
    var totalItems = 0;
    var totalPages = 0;
    var searchTerm = '';
    var statusFilter = '';
    
    // Load notifications on page load
    loadNotifications();
    
    // Apply filters
    $('#qpn_apply_filters').click(function() {
        searchTerm = $('#qpn_notification_search').val();
        statusFilter = $('#qpn_status_filter').val();
        currentPage = 1;
        loadNotifications();
    });
    
    // Reset filters
    $('#qpn_reset_filters').click(function() {
        $('#qpn_notification_search').val('');
        $('#qpn_status_filter').val('');
        searchTerm = '';
        statusFilter = '';
        currentPage = 1;
        loadNotifications();
    });
    
    // Pagination
    $('#qpn_prev_page').click(function() {
        if (currentPage > 1) {
            currentPage--;
            loadNotifications();
        }
    });
    
    $('#qpn_next_page').click(function() {
        if (currentPage < totalPages) {
            currentPage++;
            loadNotifications();
        }
    });
    
    // Modal
    $('.qpn-modal-close').click(function() {
        $('#qpn_notification_modal').hide();
    });
    
    $(window).click(function(event) {
        if ($(event.target).hasClass('qpn-modal')) {
            $('.qpn-modal').hide();
        }
    });
    
    // Load notifications function
    function loadNotifications() {
        $('#qpn_notifications_list').html('<tr class="qpn-loading-row"><td colspan="7"><?php _e('Loading notifications...', 'quick-push-notifications'); ?></td></tr>');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qpn_get_notifications',
                nonce: '<?php echo wp_create_nonce('qpn_admin_nonce'); ?>',
                offset: (currentPage - 1) * itemsPerPage,
                limit: itemsPerPage,
                search: searchTerm,
                status: statusFilter
            },
            success: function(response) {
                if (response.success) {
                    var notifications = response.data.notifications;
                    
                    // Update pagination info
                    if (notifications.length === 0) {
                        $('#qpn_pagination_info').text('<?php _e('No notifications found', 'quick-push-notifications'); ?>');
                        $('#qpn_notifications_list').html('<tr><td colspan="7"><?php _e('No notifications found.', 'quick-push-notifications'); ?></td></tr>');
                    } else {
                        totalItems = notifications.length;
                        totalPages = Math.ceil(totalItems / itemsPerPage);
                        
                        var start = (currentPage - 1) * itemsPerPage + 1;
                        var end = Math.min(start + itemsPerPage - 1, totalItems);
                        
                        $('#qpn_pagination_info').text('<?php _e('Showing', 'quick-push-notifications'); ?> ' + start + ' <?php _e('to', 'quick-push-notifications'); ?> ' + end + ' <?php _e('of', 'quick-push-notifications'); ?> ' + totalItems + ' <?php _e('notifications', 'quick-push-notifications'); ?>');
                        
                        // Update notifications list
                        var html = '';
                        $.each(notifications, function(index, notification) {
                            html += '<tr>';
                            html += '<td>' + notification.id + '</td>';
                            html += '<td>' + notification.title + '</td>';
                            html += '<td>' + truncateText(notification.message, 100) + '</td>';
                            html += '<td><span class="qpn-status qpn-status-' + notification.status + '">' + capitalizeFirstLetter(notification.status) + '</span></td>';
                            html += '<td>' + (notification.schedule_time ? formatDate(notification.schedule_time) : '-') + '</td>';
                            html += '<td>' + formatDate(notification.created_at) + '</td>';
                            html += '<td>';
                            
                            // Different actions based on status
                            if (notification.status === 'draft' || notification.status === 'scheduled') {
                                html += '<a href="<?php echo admin_url('admin.php?page=quick-push-notifications-compose&id='); ?>' + notification.id + '" class="button button-small"><?php _e('Edit', 'quick-push-notifications'); ?></a> ';
                                
                                if (notification.status === 'draft') {
                                    html += '<button class="button button-small qpn-send-notification" data-id="' + notification.id + '"><?php _e('Send', 'quick-push-notifications'); ?></button> ';
                                }
                            }
                            
                            html += '<button class="button button-small qpn-view-notification" data-id="' + notification.id + '"><?php _e('View', 'quick-push-notifications'); ?></button> ';
                            html += '<button class="button button-small qpn-delete-notification" data-id="' + notification.id + '"><?php _e('Delete', 'quick-push-notifications'); ?></button>';
                            html += '</td>';
                            html += '</tr>';
                        });
                        
                        $('#qpn_notifications_list').html(html);
                    }
                    
                    // Update pagination controls
                    $('#qpn_current_page').text(currentPage);
                    $('#qpn_prev_page').prop('disabled', currentPage === 1);
                    $('#qpn_next_page').prop('disabled', currentPage === totalPages || totalPages === 0);
                    
                    // Bind events to new buttons
                    bindNotificationActions();
                } else {
                    $('#qpn_notifications_list').html('<tr><td colspan="7"><?php _e('Error loading notifications.', 'quick-push-notifications'); ?></td></tr>');
                }
            },
            error: function() {
                $('#qpn_notifications_list').html('<tr><td colspan="7"><?php _e('Error loading notifications.', 'quick-push-notifications'); ?></td></tr>');
            }
        });
    }
    
    // Bind notification actions
    function bindNotificationActions() {
        // View notification
        $('.qpn-view-notification').click(function() {
            var notificationId = $(this).data('id');
            
            // Get notification details
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'qpn_get_notification_details',
                    nonce: '<?php echo wp_create_nonce('qpn_admin_nonce'); ?>',
                    notification_id: notificationId
                },
                success: function(response) {
                    if (response.success) {
                        var notification = response.data.notification;
                        var html = '<table class="widefat">';
                        html += '<tr><th><?php _e('ID', 'quick-push-notifications'); ?></th><td>' + notification.id + '</td></tr>';
                        html += '<tr><th><?php _e('Title', 'quick-push-notifications'); ?></th><td>' + notification.title + '</td></tr>';
                        html += '<tr><th><?php _e('Message', 'quick-push-notifications'); ?></th><td>' + notification.message + '</td></tr>';
                        html += '<tr><th><?php _e('URL', 'quick-push-notifications'); ?></th><td>' + (notification.url ? '<a href="' + notification.url + '" target="_blank">' + notification.url + '</a>' : '-') + '</td></tr>';
                        html += '<tr><th><?php _e('Status', 'quick-push-notifications'); ?></th><td><span class="qpn-status qpn-status-' + notification.status + '">' + capitalizeFirstLetter(notification.status) + '</span></td></tr>';
                        
                        if (notification.status === 'scheduled') {
                            html += '<tr><th><?php _e('Schedule Time', 'quick-push-notifications'); ?></th><td>' + formatDate(notification.schedule_time) + '</td></tr>';
                        }
                        
                        if (notification.status === 'sent') {
                            html += '<tr><th><?php _e('Sent At', 'quick-push-notifications'); ?></th><td>' + formatDate(notification.sent_at) + '</td></tr>';
                        }
                        
                        html += '<tr><th><?php _e('Created At', 'quick-push-notifications'); ?></th><td>' + formatDate(notification.created_at) + '</td></tr>';
                        html += '<tr><th><?php _e('Updated At', 'quick-push-notifications'); ?></th><td>' + formatDate(notification.updated_at) + '</td></tr>';
                        html += '</table>';
                        
                        // Add preview
                        html += '<h3><?php _e('Preview', 'quick-push-notifications'); ?></h3>';
                        html += '<div class="qpn-notification-preview">';
                        
                        if (notification.icon) {
                            html += '<div class="qpn-notification-preview-icon"><img src="' + notification.icon + '" alt="<?php _e('Icon', 'quick-push-notifications'); ?>"></div>';
                        }
                        
                        html += '<div class="qpn-notification-preview-content">';
                        html += '<div class="qpn-notification-preview-title">' + notification.title + '</div>';
                        html += '<div class="qpn-notification-preview-body">' + notification.message + '</div>';
                        html += '</div>';
                        html += '</div>';
                        
                        if (notification.image) {
                            html += '<div class="qpn-notification-preview-image"><img src="' + notification.image + '" alt="<?php _e('Notification Image', 'quick-push-notifications'); ?>"></div>';
                        }
                        
                        $('#qpn_notification_details').html(html);
                        $('#qpn_notification_modal').show();
                    } else {
                        alert(response.data.message);
                    }
                },
                error: function() {
                    alert('<?php _e('Error loading notification details.', 'quick-push-notifications'); ?>');
                }
            });
        });
        
        // Send notification
        $('.qpn-send-notification').click(function() {
            if (!confirm('<?php _e('Are you sure you want to send this notification to all subscribers?', 'quick-push-notifications'); ?>')) {
                return;
            }
            
            var notificationId = $(this).data('id');
            var button = $(this);
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'qpn_send_notification',
                    nonce: '<?php echo wp_create_nonce('qpn_admin_nonce'); ?>',
                    notification_id: notificationId
                },
                beforeSend: function() {
                    button.prop('disabled', true).text('<?php _e('Sending...', 'quick-push-notifications'); ?>');
                },
                success: function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        loadNotifications();
                    } else {
                        alert(response.data.message);
                        button.prop('disabled', false).text('<?php _e('Send', 'quick-push-notifications'); ?>');
                    }
                },
                error: function() {
                    alert('<?php _e('Error sending notification.', 'quick-push-notifications'); ?>');
                    button.prop('disabled', false).text('<?php _e('Send', 'quick-push-notifications'); ?>');
                }
            });
        });
        
        // Delete notification
        $('.qpn-delete-notification').click(function() {
            if (!confirm('<?php _e('Are you sure you want to delete this notification?', 'quick-push-notifications'); ?>')) {
                return;
            }
            
            var notificationId = $(this).data('id');
            var button = $(this);
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'qpn_delete_notification',
                    nonce: '<?php echo wp_create_nonce('qpn_admin_nonce'); ?>',
                    notification_id: notificationId
                },
                beforeSend: function() {
                    button.prop('disabled', true).text('<?php _e('Deleting...', 'quick-push-notifications'); ?>');
                },
                success: function(response) {
                    if (response.success) {
                        loadNotifications();
                    } else {
                        alert(response.data.message);
                        button.prop('disabled', false).text('<?php _e('Delete', 'quick-push-notifications'); ?>');
                    }
                },
                error: function() {
                    alert('<?php _e('Error deleting notification.', 'quick-push-notifications'); ?>');
                    button.prop('disabled', false).text('<?php _e('Delete', 'quick-push-notifications'); ?>');
                }
            });
        });
    }
    
    // Helper functions
    function formatDate(dateString) {
        if (!dateString) return '-';
        
        var date = new Date(dateString);
        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
    }
    
    function truncateText(text, maxLength) {
        if (!text) return '';
        
        if (text.length <= maxLength) {
            return text;
        }
        
        return text.substr(0, maxLength) + '...';
    }
    
    function capitalizeFirstLetter(string) {
        if (!string) return '';
        return string.charAt(0).toUpperCase() + string.slice(1);
    }
});
</script>
