<?php
/**
 * Provide a admin area view for composing notifications
 *
 * This file is used to markup the admin-facing compose page.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/admin/partials
 */

// Get notification ID if editing
$notification_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
$notification = array(
    'id' => 0,
    'title' => '',
    'message' => '',
    'url' => '',
    'icon' => get_option('qpn_notification_icon', ''),
    'image' => '',
    'status' => 'draft',
    'scheduled_at' => '',
    'target_audience' => 'all'
);

// If editing, get notification data
if ($notification_id > 0) {
    $notification_data = Quick_Push_Notifications_Notification::get_notification($notification_id);
    if ($notification_data) {
        $notification = $notification_data;
    }
}

// Get subscriber count
global $wpdb;
$subscribers_table = $wpdb->prefix . 'qpn_subscribers';
$subscriber_count = $wpdb->get_var("SELECT COUNT(*) FROM $subscribers_table");
?>

<div class="wrap">
    <h1><?php echo $notification_id > 0 ? __('Edit Notification', 'quick-push-notifications') : __('Compose Notification', 'quick-push-notifications'); ?></h1>
    
    <?php if ($subscriber_count == 0) : ?>
        <div class="notice notice-warning">
            <p><?php _e('You don\'t have any subscribers yet. Your notification will be saved but cannot be sent until you have subscribers.', 'quick-push-notifications'); ?></p>
        </div>
    <?php endif; ?>
    
    <div class="qpn-compose-form">
        <form id="qpn-notification-form">
            <input type="hidden" id="qpn_notification_id" name="id" value="<?php echo esc_attr($notification['id']); ?>">
            <input type="hidden" id="qpn_notification_nonce" name="nonce" value="<?php echo wp_create_nonce('qpn_admin_nonce'); ?>">
            
            <table class="qpn-form-table">
                <tr>
                    <th scope="row">
                        <label for="qpn_notification_title"><?php _e('Title', 'quick-push-notifications'); ?> <span class="required">*</span></label>
                    </th>
                    <td>
                        <input type="text" id="qpn_notification_title" name="title" value="<?php echo esc_attr($notification['title']); ?>" class="regular-text" required>
                        <p class="qpn-form-description"><?php _e('The title of the notification.', 'quick-push-notifications'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="qpn_notification_message"><?php _e('Message', 'quick-push-notifications'); ?> <span class="required">*</span></label>
                    </th>
                    <td>
                        <textarea id="qpn_notification_message" name="message" rows="4" class="regular-text" required><?php echo esc_textarea($notification['message']); ?></textarea>
                        <p class="qpn-form-description"><?php _e('The message body of the notification.', 'quick-push-notifications'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="qpn_notification_url"><?php _e('URL', 'quick-push-notifications'); ?></label>
                    </th>
                    <td>
                        <input type="url" id="qpn_notification_url" name="url" value="<?php echo esc_url($notification['url']); ?>" class="regular-text">
                        <p class="qpn-form-description"><?php _e('The URL to open when the notification is clicked. Leave empty to use the site homepage.', 'quick-push-notifications'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="qpn_notification_icon"><?php _e('Icon', 'quick-push-notifications'); ?></label>
                    </th>
                    <td>
                        <input type="url" id="qpn_notification_icon" name="icon" value="<?php echo esc_url($notification['icon']); ?>" class="regular-text">
                        <p class="qpn-form-description"><?php _e('URL to the notification icon. Leave empty to use the default icon.', 'quick-push-notifications'); ?></p>
                        <button type="button" class="button qpn-media-upload" data-target="qpn_notification_icon">
                            <?php _e('Select Image', 'quick-push-notifications'); ?>
                        </button>
                        <div id="qpn_notification_icon_preview" class="qpn-image-preview">
                            <?php if (!empty($notification['icon'])) : ?>
                                <img src="<?php echo esc_url($notification['icon']); ?>" alt="<?php _e('Icon Preview', 'quick-push-notifications'); ?>">
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="qpn_notification_image"><?php _e('Image', 'quick-push-notifications'); ?></label>
                    </th>
                    <td>
                        <input type="url" id="qpn_notification_image" name="image" value="<?php echo esc_url($notification['image']); ?>" class="regular-text">
                        <p class="qpn-form-description"><?php _e('URL to an image to display in the notification. Optional.', 'quick-push-notifications'); ?></p>
                        <button type="button" class="button qpn-media-upload" data-target="qpn_notification_image">
                            <?php _e('Select Image', 'quick-push-notifications'); ?>
                        </button>
                        <div id="qpn_notification_image_preview" class="qpn-image-preview">
                            <?php if (!empty($notification['image'])) : ?>
                                <img src="<?php echo esc_url($notification['image']); ?>" alt="<?php _e('Image Preview', 'quick-push-notifications'); ?>">
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="qpn_notification_status"><?php _e('Status', 'quick-push-notifications'); ?></label>
                    </th>
                    <td>
                        <select id="qpn_notification_status" name="status">
                            <option value="draft" <?php selected($notification['status'], 'draft'); ?>><?php _e('Draft', 'quick-push-notifications'); ?></option>
                            <option value="scheduled" <?php selected($notification['status'], 'scheduled'); ?>><?php _e('Scheduled', 'quick-push-notifications'); ?></option>
                        </select>
                        <p class="qpn-form-description"><?php _e('The status of the notification.', 'quick-push-notifications'); ?></p>
                    </td>
                </tr>
                <tr id="qpn_schedule_time_row" style="<?php echo $notification['status'] !== 'scheduled' ? 'display: none;' : ''; ?>">
                    <th scope="row">
                        <label for="qpn_notification_schedule_time"><?php _e('Schedule Time', 'quick-push-notifications'); ?></label>
                    </th>
                    <td>
                        <input type="datetime-local" id="qpn_notification_schedule_time" name="scheduled_at" value="<?php echo esc_attr(str_replace(' ', 'T', $notification['scheduled_at'])); ?>" class="regular-text">
                        <p class="qpn-form-description"><?php _e('The time to send the notification if scheduled.', 'quick-push-notifications'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="qpn_notification_target_audience"><?php _e('Target Audience', 'quick-push-notifications'); ?></label>
                    </th>
                    <td>
                        <select id="qpn_notification_target_audience" name="target_audience">
                            <option value="all" <?php selected($notification['target_audience'], 'all'); ?>><?php _e('All Subscribers', 'quick-push-notifications'); ?></option>
                            <!-- Additional targeting options can be added here in future versions -->
                        </select>
                        <p class="qpn-form-description"><?php _e('Select which subscribers should receive this notification.', 'quick-push-notifications'); ?></p>
                    </td>
                </tr>
            </table>
            
            <div class="qpn-compose-preview">
                <h3><?php _e('Preview', 'quick-push-notifications'); ?></h3>
                <div class="qpn-notification-preview">
                    <div id="qpn_preview_icon" class="qpn-notification-preview-icon">
                        <?php if (!empty($notification['icon'])) : ?>
                            <img src="<?php echo esc_url($notification['icon']); ?>" alt="<?php _e('Icon', 'quick-push-notifications'); ?>">
                        <?php endif; ?>
                    </div>
                    <div class="qpn-notification-preview-content">
                        <div id="qpn_preview_title" class="qpn-notification-preview-title"><?php echo esc_html($notification['title'] ? $notification['title'] : __('Notification Title', 'quick-push-notifications')); ?></div>
                        <div id="qpn_preview_body" class="qpn-notification-preview-body"><?php echo esc_html($notification['message'] ? $notification['message'] : __('Notification message will appear here.', 'quick-push-notifications')); ?></div>
                    </div>
                </div>
                <?php if (!empty($notification['image'])) : ?>
                    <div id="qpn_preview_image" class="qpn-notification-preview-image">
                        <img src="<?php echo esc_url($notification['image']); ?>" alt="<?php _e('Notification Image', 'quick-push-notifications'); ?>">
                    </div>
                <?php endif; ?>
                <p class="qpn-preview-note"><?php _e('Note: Actual notification appearance may vary depending on the browser and device.', 'quick-push-notifications'); ?></p>
            </div>
            
            <div class="qpn-form-actions">
                <button type="submit" id="qpn_save_notification" class="button button-primary">
                    <?php echo $notification_id > 0 ? __('Update Notification', 'quick-push-notifications') : __('Save Notification', 'quick-push-notifications'); ?>
                </button>
                
                <?php if ($notification_id > 0 && $subscriber_count > 0) : ?>
                    <button type="button" id="qpn_send_notification" class="button" data-id="<?php echo esc_attr($notification_id); ?>">
                        <?php _e('Send Now', 'quick-push-notifications'); ?>
                    </button>
                <?php elseif ($subscriber_count > 0) : ?>
                    <button type="button" id="qpn_save_and_send" class="button">
                        <?php _e('Save & Send', 'quick-push-notifications'); ?>
                    </button>
                <?php endif; ?>
                
                <a href="<?php echo admin_url('admin.php?page=quick-push-notifications-notifications'); ?>" class="button">
                    <?php _e('Cancel', 'quick-push-notifications'); ?>
                </a>
            </div>
        </form>
    </div>
</div>

<script type="text/javascript">
jQuery(document).ready(function($) {
    // Media uploader
    $('.qpn-media-upload').click(function(e) {
        e.preventDefault();
        
        var button = $(this);
        var targetField = button.data('target');
        
        var mediaUploader = wp.media({
            title: '<?php _e('Select Image', 'quick-push-notifications'); ?>',
            button: {
                text: '<?php _e('Use this image', 'quick-push-notifications'); ?>'
            },
            multiple: false
        });
        
        mediaUploader.on('select', function() {
            var attachment = mediaUploader.state().get('selection').first().toJSON();
            $('#' + targetField).val(attachment.url);
            $('#' + targetField + '_preview').html('<img src="' + attachment.url + '" alt="<?php _e('Preview', 'quick-push-notifications'); ?>">');
            
            // Update preview
            if (targetField === 'qpn_notification_icon') {
                $('#qpn_preview_icon').html('<img src="' + attachment.url + '" alt="<?php _e('Icon', 'quick-push-notifications'); ?>">');
            } else if (targetField === 'qpn_notification_image') {
                $('#qpn_preview_image').html('<img src="' + attachment.url + '" alt="<?php _e('Image', 'quick-push-notifications'); ?>">');
            }
        });
        
        mediaUploader.open();
    });
    
    // Show/hide schedule time field
    $('#qpn_notification_status').change(function() {
        if ($(this).val() === 'scheduled') {
            $('#qpn_schedule_time_row').show();
        } else {
            $('#qpn_schedule_time_row').hide();
        }
    });
    
    // Live preview
    $('#qpn_notification_title').keyup(function() {
        $('#qpn_preview_title').text($(this).val() || '<?php _e('Notification Title', 'quick-push-notifications'); ?>');
    });
    
    $('#qpn_notification_message').keyup(function() {
        $('#qpn_preview_body').text($(this).val() || '<?php _e('Notification message will appear here.', 'quick-push-notifications'); ?>');
    });
    
    // Save notification
    $('#qpn-notification-form').submit(function(e) {
        e.preventDefault();
        
        var formData = $(this).serialize();
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qpn_save_notification',
                ...formData
            },
            beforeSend: function() {
                $('#qpn_save_notification').prop('disabled', true).text('<?php _e('Saving...', 'quick-push-notifications'); ?>');
            },
            success: function(response) {
                if (response.success) {
                    // If new notification, redirect to edit page
                    if ($('#qpn_notification_id').val() === '0') {
                        window.location.href = '<?php echo admin_url('admin.php?page=quick-push-notifications-compose&id='); ?>' + response.data.notification_id + '&saved=1';
                    } else {
                        alert(response.data.message);
                        $('#qpn_save_notification').prop('disabled', false).text('<?php _e('Update Notification', 'quick-push-notifications'); ?>');
                    }
                } else {
                    alert(response.data.message);
                    $('#qpn_save_notification').prop('disabled', false).text('<?php echo $notification_id > 0 ? __('Update Notification', 'quick-push-notifications') : __('Save Notification', 'quick-push-notifications'); ?>');
                }
            },
            error: function() {
                alert('<?php _e('An error occurred while saving the notification.', 'quick-push-notifications'); ?>');
                $('#qpn_save_notification').prop('disabled', false).text('<?php echo $notification_id > 0 ? __('Update Notification', 'quick-push-notifications') : __('Save Notification', 'quick-push-notifications'); ?>');
            }
        });
    });
    
    // Send notification
    $('#qpn_send_notification').click(function() {
        if (!confirm('<?php _e('Are you sure you want to send this notification to all subscribers?', 'quick-push-notifications'); ?>')) {
            return;
        }
        
        var notificationId = $(this).data('id');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qpn_send_notification',
                notification_id: notificationId,
                nonce: $('#qpn_notification_nonce').val()
            },
            beforeSend: function() {
                $('#qpn_send_notification').prop('disabled', true).text('<?php _e('Sending...', 'quick-push-notifications'); ?>');
            },
            success: function(response) {
                if (response.success) {
                    alert(response.data.message);
                    window.location.href = '<?php echo admin_url('admin.php?page=quick-push-notifications-notifications'); ?>';
                } else {
                    alert(response.data.message);
                    $('#qpn_send_notification').prop('disabled', false).text('<?php _e('Send Now', 'quick-push-notifications'); ?>');
                }
            },
            error: function() {
                alert('<?php _e('An error occurred while sending the notification.', 'quick-push-notifications'); ?>');
                $('#qpn_send_notification').prop('disabled', false).text('<?php _e('Send Now', 'quick-push-notifications'); ?>');
            }
        });
    });
    
    // Save and send
    $('#qpn_save_and_send').click(function() {
        if (!confirm('<?php _e('Are you sure you want to save and send this notification to all subscribers?', 'quick-push-notifications'); ?>')) {
            return;
        }
        
        var formData = $('#qpn-notification-form').serialize();
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qpn_save_notification',
                ...formData
            },
            beforeSend: function() {
                $('#qpn_save_and_send').prop('disabled', true).text('<?php _e('Saving...', 'quick-push-notifications'); ?>');
            },
            success: function(response) {
                if (response.success) {
                    var notificationId = response.data.notification_id;
                    
                    // Now send the notification
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'qpn_send_notification',
                            notification_id: notificationId,
                            nonce: $('#qpn_notification_nonce').val()
                        },
                        beforeSend: function() {
                            $('#qpn_save_and_send').text('<?php _e('Sending...', 'quick-push-notifications'); ?>');
                        },
                        success: function(sendResponse) {
                            if (sendResponse.success) {
                                alert(sendResponse.data.message);
                                window.location.href = '<?php echo admin_url('admin.php?page=quick-push-notifications-notifications'); ?>';
                            } else {
                                alert(sendResponse.data.message);
                                $('#qpn_save_and_send').prop('disabled', false).text('<?php _e('Save & Send', 'quick-push-notifications'); ?>');
                            }
                        },
                        error: function() {
                            alert('<?php _e('An error occurred while sending the notification.', 'quick-push-notifications'); ?>');
                            $('#qpn_save_and_send').prop('disabled', false).text('<?php _e('Save & Send', 'quick-push-notifications'); ?>');
                        }
                    });
                } else {
                    alert(response.data.message);
                    $('#qpn_save_and_send').prop('disabled', false).text('<?php _e('Save & Send', 'quick-push-notifications'); ?>');
                }
            },
            error: function() {
                alert('<?php _e('An error occurred while saving the notification.', 'quick-push-notifications'); ?>');
                $('#qpn_save_and_send').prop('disabled', false).text('<?php _e('Save & Send', 'quick-push-notifications'); ?>');
            }
        });
    });
});
</script>
