<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://plugins.quickpr.co/
 * @since      1.0.0
 *
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two hooks for
 * enqueuing the admin-specific stylesheet and JavaScript.
 *
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/admin
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Push_Notifications_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/quick-push-notifications-admin.css', array(), $this->version, 'all');
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/quick-push-notifications-admin.js', array('jquery'), $this->version, false);
        
        wp_localize_script($this->plugin_name, 'qpn_admin_vars', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('qpn_admin_nonce'),
            'site_url' => site_url(),
            'firebase_config' => Quick_Push_Notifications_Firebase::get_firebase_config()
        ));
    }

    /**
     * Add menu items to the admin menu.
     *
     * @since    1.0.0
     */
    public function add_plugin_admin_menu() {
        // Main menu item
        add_menu_page(
            __('Quick Push Notifications', 'quick-push-notifications'),
            __('Push Notifications', 'quick-push-notifications'),
            'manage_options',
            $this->plugin_name,
            array($this, 'display_plugin_admin_dashboard'),
            'dashicons-megaphone',
            26
        );
        
        // Dashboard submenu
        add_submenu_page(
            $this->plugin_name,
            __('Dashboard', 'quick-push-notifications'),
            __('Dashboard', 'quick-push-notifications'),
            'manage_options',
            $this->plugin_name,
            array($this, 'display_plugin_admin_dashboard')
        );
        
        // Subscribers submenu
        add_submenu_page(
            $this->plugin_name,
            __('Subscribers', 'quick-push-notifications'),
            __('Subscribers', 'quick-push-notifications'),
            'manage_options',
            $this->plugin_name . '-subscribers',
            array($this, 'display_plugin_admin_subscribers')
        );
        
        // Notifications submenu
        add_submenu_page(
            $this->plugin_name,
            __('Notifications', 'quick-push-notifications'),
            __('Notifications', 'quick-push-notifications'),
            'manage_options',
            $this->plugin_name . '-notifications',
            array($this, 'display_plugin_admin_notifications')
        );
        
        // Compose submenu
        add_submenu_page(
            $this->plugin_name,
            __('Compose', 'quick-push-notifications'),
            __('Compose', 'quick-push-notifications'),
            'manage_options',
            $this->plugin_name . '-compose',
            array($this, 'display_plugin_admin_compose')
        );
        
        // Settings submenu
        add_submenu_page(
            $this->plugin_name,
            __('Settings', 'quick-push-notifications'),
            __('Settings', 'quick-push-notifications'),
            'manage_options',
            $this->plugin_name . '-settings',
            array($this, 'display_plugin_admin_settings')
        );
    }

    /**
     * Add action links to the plugin list.
     *
     * @since    1.0.0
     * @param    array    $links    The existing action links.
     * @return   array              The modified action links.
     */
    public function add_action_links($links) {
        $settings_link = array(
            '<a href="' . admin_url('admin.php?page=' . $this->plugin_name . '-settings') . '">' . __('Settings', 'quick-push-notifications') . '</a>',
        );
        return array_merge($settings_link, $links);
    }

    /**
     * Register plugin settings.
     *
     * @since    1.0.0
     */
    public function register_settings() {
        // Firebase settings
        register_setting($this->plugin_name . '-settings', 'qpn_firebase_api_key');
        register_setting($this->plugin_name . '-settings', 'qpn_firebase_auth_domain');
        register_setting($this->plugin_name . '-settings', 'qpn_firebase_project_id');
        register_setting($this->plugin_name . '-settings', 'qpn_firebase_storage_bucket');
        register_setting($this->plugin_name . '-settings', 'qpn_firebase_messaging_sender_id');
        register_setting($this->plugin_name . '-settings', 'qpn_firebase_app_id');
        register_setting($this->plugin_name . '-settings', 'qpn_firebase_server_key');
        
        // Notification settings
        register_setting($this->plugin_name . '-settings', 'qpn_notification_icon');
        register_setting($this->plugin_name . '-settings', 'qpn_welcome_notification_enabled');
        register_setting($this->plugin_name . '-settings', 'qpn_welcome_notification_title');
        register_setting($this->plugin_name . '-settings', 'qpn_welcome_notification_message');
        
        // Auto-push settings
        register_setting($this->plugin_name . '-settings', 'qpn_auto_push_enabled');
        register_setting($this->plugin_name . '-settings', 'qpn_auto_push_post_types');
        register_setting($this->plugin_name . '-settings', 'qpn_auto_push_title');
        register_setting($this->plugin_name . '-settings', 'qpn_auto_push_message');
        register_setting($this->plugin_name . '-settings', 'qpn_auto_push_send');
    }

    /**
     * Display the dashboard page.
     *
     * @since    1.0.0
     */
    public function display_plugin_admin_dashboard() {
        include_once 'partials/quick-push-notifications-admin-dashboard.php';
    }

    /**
     * Display the subscribers page.
     *
     * @since    1.0.0
     */
    public function display_plugin_admin_subscribers() {
        include_once 'partials/quick-push-notifications-admin-subscribers.php';
    }

    /**
     * Display the notifications page.
     *
     * @since    1.0.0
     */
    public function display_plugin_admin_notifications() {
        include_once 'partials/quick-push-notifications-admin-notifications.php';
    }

    /**
     * Display the compose page.
     *
     * @since    1.0.0
     */
    public function display_plugin_admin_compose() {
        include_once 'partials/quick-push-notifications-admin-compose.php';
    }

    /**
     * Display the settings page.
     *
     * @since    1.0.0
     */
    public function display_plugin_admin_settings() {
        include_once 'partials/quick-push-notifications-admin-settings.php';
    }

    /**
     * AJAX handler for sending test notifications.
     *
     * @since    1.0.0
     */
    public function ajax_send_test_notification() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qpn_admin_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'You do not have permission to perform this action'));
        }
        
        // Get notification data
        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : 'Test Notification';
        $message = isset($_POST['message']) ? sanitize_text_field($_POST['message']) : 'This is a test notification from your website.';
        $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : home_url();
        $icon = isset($_POST['icon']) ? esc_url_raw($_POST['icon']) : '';
        
        // Create notification data
        $notification = array(
            'title' => $title,
            'message' => $message,
            'url' => $url,
            'icon' => $icon
        );
        
        // Send to all subscribers
        $result = Quick_Push_Notifications_Firebase::send_notification_to_all($notification);
        
        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        } else {
            wp_send_json_success(array(
                'message' => 'Test notification sent successfully',
                'result' => $result
            ));
        }
    }

    /**
     * AJAX handler for getting subscribers.
     *
     * @since    1.0.0
     */
    public function ajax_get_subscribers() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qpn_admin_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'You do not have permission to perform this action'));
        }
        
        // Get query parameters
        $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 10;
        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        $search = isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '';
        $browser = isset($_POST['browser']) ? sanitize_text_field($_POST['browser']) : '';
        $os = isset($_POST['os']) ? sanitize_text_field($_POST['os']) : '';
        $device_type = isset($_POST['device_type']) ? sanitize_text_field($_POST['device_type']) : '';
        
        // Get subscribers
        $subscribers = Quick_Push_Notifications_Subscriber::get_subscribers(array(
            'limit' => $limit,
            'offset' => $offset,
            'search' => $search,
            'browser' => $browser,
            'os' => $os,
            'device_type' => $device_type
        ));
        
        // Get total count
        $total = Quick_Push_Notifications_Subscriber::count_subscribers(array(
            'search' => $search,
            'browser' => $browser,
            'os' => $os,
            'device_type' => $device_type
        ));
        
        wp_send_json_success(array(
            'subscribers' => $subscribers,
            'total' => $total
        ));
    }

    /**
     * AJAX handler for deleting subscribers.
     *
     * @since    1.0.0
     */
    public function ajax_delete_subscriber() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qpn_admin_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'You do not have permission to perform this action'));
        }
        
        // Get subscriber ID
        $subscriber_id = isset($_POST['subscriber_id']) ? intval($_POST['subscriber_id']) : 0;
        
        if ($subscriber_id <= 0) {
            wp_send_json_error(array('message' => 'Invalid subscriber ID'));
        }
        
        // Delete subscriber
        $result = Quick_Push_Notifications_Subscriber::delete_subscriber($subscriber_id);
        
        if ($result) {
            wp_send_json_success(array('message' => 'Subscriber deleted successfully'));
        } else {
            wp_send_json_error(array('message' => 'Failed to delete subscriber'));
        }
    }

    /**
     * AJAX handler for getting notifications.
     *
     * @since    1.0.0
     */
    public function ajax_get_notifications() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qpn_admin_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'You do not have permission to perform this action'));
        }
        
        // Get query parameters
        $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 10;
        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        $search = isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '';
        $status = isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '';
        
        // Get notifications
        $notifications = Quick_Push_Notifications_Notification::get_notifications(array(
            'limit' => $limit,
            'offset' => $offset,
            'search' => $search,
            'status' => $status
        ));
        
        wp_send_json_success(array(
            'notifications' => $notifications
        ));
    }

    /**
     * AJAX handler for saving notifications.
     *
     * @since    1.0.0
     */
    public function ajax_save_notification() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qpn_admin_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'You do not have permission to perform this action'));
        }
        
        // Get notification data
        $notification_id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
        $message = isset($_POST['message']) ? wp_kses_post($_POST['message']) : '';
        $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';
        $icon = isset($_POST['icon']) ? esc_url_raw($_POST['icon']) : '';
        $image = isset($_POST['image']) ? esc_url_raw($_POST['image']) : '';
        $status = isset($_POST['status']) ? sanitize_text_field($_POST['status']) : 'draft';
        $scheduled_at = isset($_POST['scheduled_at']) ? sanitize_text_field($_POST['scheduled_at']) : '';
        $target_audience = isset($_POST['target_audience']) ? sanitize_text_field($_POST['target_audience']) : 'all';
        
        // Validate required fields
        if (empty($title) || empty($message)) {
            wp_send_json_error(array('message' => 'Title and message are required'));
        }
        
        // Create notification data
        $notification_data = array(
            'id' => $notification_id,
            'title' => $title,
            'message' => $message,
            'url' => $url,
            'icon' => $icon,
            'image' => $image,
            'status' => $status,
            'scheduled_at' => $scheduled_at,
            'target_audience' => $target_audience
        );
        
        // Save notification
        $result = Quick_Push_Notifications_Notification::save_notification($notification_data);
        
        if ($result) {
            wp_send_json_success(array(
                'message' => 'Notification saved successfully',
                'notification_id' => $result
            ));
        } else {
            wp_send_json_error(array('message' => 'Failed to save notification'));
        }
    }

    /**
     * AJAX handler for sending notifications.
     *
     * @since    1.0.0
     */
    public function ajax_send_notification() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qpn_admin_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'You do not have permission to perform this action'));
        }
        
        // Get notification ID
        $notification_id = isset($_POST['notification_id']) ? intval($_POST['notification_id']) : 0;
        
        if ($notification_id <= 0) {
            wp_send_json_error(array('message' => 'Invalid notification ID'));
        }
        
        // Send notification
        $result = Quick_Push_Notifications_Notification::send_notification($notification_id);
        
        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        } else {
            wp_send_json_success(array(
                'message' => 'Notification sent successfully',
                'result' => $result
            ));
        }
    }

    /**
     * AJAX handler for deleting notifications.
     *
     * @since    1.0.0
     */
    public function ajax_delete_notification() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qpn_admin_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'You do not have permission to perform this action'));
        }
        
        // Get notification ID
        $notification_id = isset($_POST['notification_id']) ? intval($_POST['notification_id']) : 0;
        
        if ($notification_id <= 0) {
            wp_send_json_error(array('message' => 'Invalid notification ID'));
        }
        
        // Delete notification
        $result = Quick_Push_Notifications_Notification::delete_notification($notification_id);
        
        if ($result) {
            wp_send_json_success(array('message' => 'Notification deleted successfully'));
        } else {
            wp_send_json_error(array('message' => 'Failed to delete notification'));
        }
    }

    /**
     * Check post status for auto-push notification.
     *
     * @since    1.0.0
     * @param    string     $new_status    The new post status.
     * @param    string     $old_status    The old post status.
     * @param    WP_Post    $post          The post object.
     */
    public function check_post_status_for_notification($new_status, $old_status, $post) {
        // Check if auto-push is enabled
        if (get_option('qpn_auto_push_enabled', 'no') !== 'yes') {
            return;
        }
        
        // Only proceed if the post is being published for the first time
        if ($new_status === 'publish' && $old_status !== 'publish') {
            Quick_Push_Notifications_Notification::create_post_notification($post->ID);
        }
    }

    /**
     * Send scheduled notifications.
     *
     * @since    1.0.0
     */
    public function send_scheduled_notifications() {
        Quick_Push_Notifications_Notification::process_scheduled_notifications();
    }
}
