<?php
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://plugins.quickpr.co/
 * @since      1.0.0
 *
 * @package    Quick_Popup_Builder
 * @subpackage Quick_Popup_Builder/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Quick_Popup_Builder
 * @subpackage Quick_Popup_Builder/includes
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Popup_Builder {

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Quick_Popup_Builder_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if (defined('QUICK_POPUP_BUILDER_VERSION')) {
            $this->version = QUICK_POPUP_BUILDER_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'quick-popup-builder';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - Quick_Popup_Builder_Loader. Orchestrates the hooks of the plugin.
     * - Quick_Popup_Builder_i18n. Defines internationalization functionality.
     * - Quick_Popup_Builder_Admin. Defines all hooks for the admin area.
     * - Quick_Popup_Builder_Public. Defines all hooks for the public side of the site.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-quick-popup-builder-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-quick-popup-builder-i18n.php';

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-quick-popup-builder-admin.php';

        /**
         * The class responsible for defining all actions that occur in the public-facing
         * side of the site.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'public/class-quick-popup-builder-public.php';

        $this->loader = new Quick_Popup_Builder_Loader();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Quick_Popup_Builder_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {
        $plugin_i18n = new Quick_Popup_Builder_i18n();
        $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {
        $plugin_admin = new Quick_Popup_Builder_Admin($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');
        
        // Add menu items
        $this->loader->add_action('admin_menu', $plugin_admin, 'add_plugin_admin_menu');
        
        // Add Settings link to the plugin
        $this->loader->add_filter('plugin_action_links_' . plugin_basename(plugin_dir_path(dirname(__FILE__)) . $this->plugin_name . '.php'), $plugin_admin, 'add_action_links');
        
        // Register post type
        $this->loader->add_action('init', $plugin_admin, 'register_popup_post_type');
        
        // Add meta boxes
        $this->loader->add_action('add_meta_boxes', $plugin_admin, 'add_popup_meta_boxes');
        
        // Save post meta
        $this->loader->add_action('save_post', $plugin_admin, 'save_popup_meta', 10, 2);
        
        // AJAX handlers
        $this->loader->add_action('wp_ajax_quick_popup_get_stats', $plugin_admin, 'get_popup_stats');
        $this->loader->add_action('wp_ajax_quick_popup_reset_stats', $plugin_admin, 'reset_popup_stats');
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks() {
        $plugin_public = new Quick_Popup_Builder_Public($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_styles');
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_scripts');
        
        // Display popups
        $this->loader->add_action('wp_footer', $plugin_public, 'display_popups');
        
        // AJAX handlers for tracking
        $this->loader->add_action('wp_ajax_quick_popup_track', $plugin_public, 'track_popup_event');
        $this->loader->add_action('wp_ajax_nopriv_quick_popup_track', $plugin_public, 'track_popup_event');
        
        // Shortcode for inline popups
        $this->loader->add_shortcode('quick_popup', $plugin_public, 'popup_shortcode');
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    Quick_Popup_Builder_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }
}
