<?php
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://plugins.quickpr.co/
 * @since      1.0.0
 *
 * @package    Quick_Image_Optimizer
 * @subpackage Quick_Image_Optimizer/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Quick_Image_Optimizer
 * @subpackage Quick_Image_Optimizer/includes
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Image_Optimizer {

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Quick_Image_Optimizer_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if (defined('QUICK_IMAGE_OPTIMIZER_VERSION')) {
            $this->version = QUICK_IMAGE_OPTIMIZER_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'quick-image-optimizer';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - Quick_Image_Optimizer_Loader. Orchestrates the hooks of the plugin.
     * - Quick_Image_Optimizer_i18n. Defines internationalization functionality.
     * - Quick_Image_Optimizer_Admin. Defines all hooks for the admin area.
     * - Quick_Image_Optimizer_Public. Defines all hooks for the public side of the site.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-quick-image-optimizer-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-quick-image-optimizer-i18n.php';

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-quick-image-optimizer-admin.php';

        /**
         * The class responsible for defining all actions that occur in the public-facing
         * side of the site.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'public/class-quick-image-optimizer-public.php';

        /**
         * The class responsible for handling image optimization.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-quick-image-optimizer-engine.php';

        /**
         * The class responsible for WebP conversion.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-quick-image-optimizer-webp.php';

        /**
         * The class responsible for lazy loading.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-quick-image-optimizer-lazy-load.php';

        /**
         * The class responsible for CDN integration.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-quick-image-optimizer-cdn.php';

        $this->loader = new Quick_Image_Optimizer_Loader();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Quick_Image_Optimizer_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {
        $plugin_i18n = new Quick_Image_Optimizer_i18n();
        $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {
        $plugin_admin = new Quick_Image_Optimizer_Admin($this->get_plugin_name(), $this->get_version());

        // Enqueue styles and scripts
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');
        
        // Add menu items
        $this->loader->add_action('admin_menu', $plugin_admin, 'add_plugin_admin_menu');
        
        // Add Settings link to the plugin
        $this->loader->add_filter('plugin_action_links_' . plugin_basename(plugin_dir_path(dirname(__FILE__)) . $this->plugin_name . '.php'), $plugin_admin, 'add_action_links');
        
        // Register settings
        $this->loader->add_action('admin_init', $plugin_admin, 'register_settings');
        
        // Add media library columns
        $this->loader->add_filter('manage_media_columns', $plugin_admin, 'add_media_columns');
        $this->loader->add_action('manage_media_custom_column', $plugin_admin, 'display_media_column_content', 10, 2);
        
        // Add bulk optimization action
        $this->loader->add_filter('bulk_actions-upload', $plugin_admin, 'add_bulk_actions');
        $this->loader->add_filter('handle_bulk_actions-upload', $plugin_admin, 'handle_bulk_actions', 10, 3);
        
        // Add optimization on upload
        $this->loader->add_action('add_attachment', $plugin_admin, 'optimize_on_upload');
        
        // Add AJAX actions
        $this->loader->add_action('wp_ajax_qio_optimize_image', $plugin_admin, 'ajax_optimize_image');
        $this->loader->add_action('wp_ajax_qio_restore_image', $plugin_admin, 'ajax_restore_image');
        $this->loader->add_action('wp_ajax_qio_bulk_optimize', $plugin_admin, 'ajax_bulk_optimize');
        
        // Add dashboard widget
        $this->loader->add_action('wp_dashboard_setup', $plugin_admin, 'add_dashboard_widget');
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks() {
        $plugin_public = new Quick_Image_Optimizer_Public($this->get_plugin_name(), $this->get_version());
        $lazy_load = new Quick_Image_Optimizer_Lazy_Load($this->get_plugin_name(), $this->get_version());
        $cdn = new Quick_Image_Optimizer_CDN($this->get_plugin_name(), $this->get_version());

        // Enqueue styles and scripts
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_styles');
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_scripts');
        
        // Add WebP support
        $this->loader->add_filter('wp_get_attachment_image_src', $plugin_public, 'maybe_use_webp', 10, 4);
        $this->loader->add_filter('wp_calculate_image_srcset', $plugin_public, 'maybe_use_webp_srcset', 10, 5);
        
        // Add lazy loading if enabled
        if (get_option('qio_enable_lazy_loading', '1')) {
            $this->loader->add_filter('the_content', $lazy_load, 'add_lazy_loading', 99);
            $this->loader->add_filter('post_thumbnail_html', $lazy_load, 'add_lazy_loading', 99);
            $this->loader->add_filter('widget_text', $lazy_load, 'add_lazy_loading', 99);
        }
        
        // Add CDN support if enabled
        if (get_option('qio_enable_cdn', '0')) {
            $this->loader->add_filter('wp_get_attachment_url', $cdn, 'rewrite_attachment_url', 10, 2);
            $this->loader->add_filter('the_content', $cdn, 'rewrite_image_urls', 100);
        }
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    Quick_Image_Optimizer_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }
}
