<?php
/**
 * Fired during plugin activation
 *
 * @link       https://plugins.quickpr.co/
 * @since      1.0.0
 *
 * @package    Quick_Image_Optimizer
 * @subpackage Quick_Image_Optimizer/includes
 */

/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since      1.0.0
 * @package    Quick_Image_Optimizer
 * @subpackage Quick_Image_Optimizer/includes
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Image_Optimizer_Activator {

    /**
     * Set up plugin options and create necessary directories on activation.
     *
     * @since    1.0.0
     */
    public static function activate() {
        // Default plugin options
        $default_options = array(
            'compression_level' => 'medium', // Options: low, medium, high
            'max_width' => 1920,
            'max_height' => 1080,
            'preserve_exif' => false,
            'optimize_on_upload' => true,
            'enable_webp' => true,
            'enable_lazy_loading' => true,
            'enable_cdn' => false,
            'cdn_url' => '',
            'cdn_directories' => 'wp-content/uploads',
            'exclude_sizes' => array(),
            'exclude_paths' => array(),
        );
        
        // Only add options if they don't exist
        if (!get_option('quick_image_optimizer_options')) {
            add_option('quick_image_optimizer_options', $default_options);
        }
        
        // Create backup directory
        self::create_backup_directory();
        
        // Create database tables if needed
        self::create_tables();
        
        // Check for optimization libraries
        self::check_optimization_libraries();
    }
    
    /**
     * Create directory for storing original image backups.
     *
     * @since    1.0.0
     */
    private static function create_backup_directory() {
        $upload_dir = wp_upload_dir();
        $backup_dir = $upload_dir['basedir'] . '/qio-backups';
        
        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);
        }
        
        // Create an empty index.php file to prevent directory listing
        if (!file_exists($backup_dir . '/index.php')) {
            $index_file = fopen($backup_dir . '/index.php', 'w');
            fwrite($index_file, '<?php // Silence is golden');
            fclose($index_file);
        }
        
        // Create .htaccess to protect the backup directory
        if (!file_exists($backup_dir . '/.htaccess')) {
            $htaccess_file = fopen($backup_dir . '/.htaccess', 'w');
            fwrite($htaccess_file, "Order Allow,Deny\nDeny from all");
            fclose($htaccess_file);
        }
    }
    
    /**
     * Create custom database tables for tracking image optimization.
     *
     * @since    1.0.0
     */
    private static function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        $table_name = $wpdb->prefix . 'qio_images';
        
        $sql = "CREATE TABLE $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            attachment_id bigint(20) NOT NULL,
            file_path varchar(255) NOT NULL,
            original_size bigint(20) NOT NULL DEFAULT 0,
            optimized_size bigint(20) NOT NULL DEFAULT 0,
            savings_percentage float NOT NULL DEFAULT 0,
            optimization_date datetime DEFAULT NULL,
            webp_size bigint(20) NOT NULL DEFAULT 0,
            webp_path varchar(255) DEFAULT NULL,
            optimization_level varchar(20) DEFAULT NULL,
            status varchar(20) NOT NULL DEFAULT 'pending',
            error_message text,
            PRIMARY KEY  (id),
            KEY attachment_id (attachment_id),
            KEY status (status)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Check for optimization libraries and set availability flags.
     *
     * @since    1.0.0
     */
    private static function check_optimization_libraries() {
        $libraries = array(
            'gd' => extension_loaded('gd'),
            'imagick' => extension_loaded('imagick'),
            'optipng' => self::command_exists('optipng'),
            'jpegoptim' => self::command_exists('jpegoptim'),
            'cwebp' => self::command_exists('cwebp'),
        );
        
        update_option('qio_available_libraries', $libraries);
    }
    
    /**
     * Check if a command exists on the system.
     *
     * @since    1.0.0
     * @param    string    $command    The command to check.
     * @return   boolean               Whether the command exists.
     */
    private static function command_exists($command) {
        $command = escapeshellcmd($command);
        $whereIsCommand = (PHP_OS == 'WINNT') ? 'where' : 'which';
        
        $process = proc_open(
            "$whereIsCommand $command",
            array(
                0 => array("pipe", "r"),
                1 => array("pipe", "w"),
                2 => array("pipe", "w"),
            ),
            $pipes
        );
        
        if ($process !== false) {
            $stdout = stream_get_contents($pipes[1]);
            $stderr = stream_get_contents($pipes[2]);
            fclose($pipes[1]);
            fclose($pipes[2]);
            proc_close($process);
            
            return $stdout != '';
        }
        
        return false;
    }
}
