<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @since      1.0.0
 */
class Quick_Event_Management_Public {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/quick-event-management-public.css', array(), $this->version, 'all');
    }

    /**
     * Register the JavaScript for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/quick-event-management-public.js', array('jquery'), $this->version, false);
        
        // Localize script for AJAX
        wp_localize_script($this->plugin_name, 'qem_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('qem_ajax_nonce'),
        ));
    }

    /**
     * Register shortcodes.
     *
     * @since    1.0.0
     */
    public function register_shortcodes() {
        add_shortcode('qem_events', array($this, 'events_shortcode'));
        add_shortcode('qem_event', array($this, 'event_shortcode'));
        add_shortcode('qem_registration_form', array($this, 'registration_form_shortcode'));
        add_shortcode('qem_calendar', array($this, 'calendar_shortcode'));
    }

    /**
     * Shortcode for displaying events.
     *
     * @since    1.0.0
     * @param    array    $atts    Shortcode attributes.
     * @return   string   Shortcode output.
     */
    public function events_shortcode($atts) {
        $atts = shortcode_atts(array(
            'limit' => 10,
            'category' => '',
            'tag' => '',
            'date_filter' => 'upcoming', // upcoming, past, all
            'order' => 'ASC',
            'layout' => 'list', // list, grid
        ), $atts, 'qem_events');
        
        // Build query args
        $args = array(
            'posts_per_page' => $atts['limit'],
            'date_filter' => $atts['date_filter'],
            'order' => $atts['order'],
        );
        
        // Add category filter
        if (!empty($atts['category'])) {
            $args['tax_query'][] = array(
                'taxonomy' => 'qem_event_category',
                'field' => 'slug',
                'terms' => explode(',', $atts['category']),
            );
        }
        
        // Add tag filter
        if (!empty($atts['tag'])) {
            $args['tax_query'][] = array(
                'taxonomy' => 'qem_event_tag',
                'field' => 'slug',
                'terms' => explode(',', $atts['tag']),
            );
        }
        
        // Get events
        $events = Quick_Event_Management_Events::get_events($args);
        
        // Start output buffering
        ob_start();
        
        // Include template based on layout
        if ($atts['layout'] === 'grid') {
            include plugin_dir_path(__FILE__) . 'partials/quick-event-management-events-grid.php';
        } else {
            include plugin_dir_path(__FILE__) . 'partials/quick-event-management-events-list.php';
        }
        
        // Return the buffered content
        return ob_get_clean();
    }

    /**
     * Shortcode for displaying a single event.
     *
     * @since    1.0.0
     * @param    array    $atts    Shortcode attributes.
     * @return   string   Shortcode output.
     */
    public function event_shortcode($atts) {
        $atts = shortcode_atts(array(
            'id' => 0,
            'show_registration' => 'yes',
        ), $atts, 'qem_event');
        
        // If no ID provided, try to get current post
        $event_id = $atts['id'];
        if (!$event_id) {
            global $post;
            if ($post && $post->post_type === 'qem_event') {
                $event_id = $post->ID;
            }
        }
        
        // If still no ID, return error message
        if (!$event_id) {
            return '<p>' . __('Event ID not provided.', 'quick-event-management') . '</p>';
        }
        
        // Get event details
        $event = Quick_Event_Management_Events::get_event_details($event_id);
        
        // If event not found, return error message
        if (!$event) {
            return '<p>' . __('Event not found.', 'quick-event-management') . '</p>';
        }
        
        // Start output buffering
        ob_start();
        
        // Include template
        include plugin_dir_path(__FILE__) . 'partials/quick-event-management-single-event.php';
        
        // Return the buffered content
        return ob_get_clean();
    }

    /**
     * Shortcode for displaying registration form.
     *
     * @since    1.0.0
     * @param    array    $atts    Shortcode attributes.
     * @return   string   Shortcode output.
     */
    public function registration_form_shortcode($atts) {
        $atts = shortcode_atts(array(
            'id' => 0,
        ), $atts, 'qem_registration_form');
        
        // If no ID provided, try to get current post
        $event_id = $atts['id'];
        if (!$event_id) {
            global $post;
            if ($post && $post->post_type === 'qem_event') {
                $event_id = $post->ID;
            }
        }
        
        // If still no ID, return error message
        if (!$event_id) {
            return '<p>' . __('Event ID not provided.', 'quick-event-management') . '</p>';
        }
        
        // Get event details
        $event = Quick_Event_Management_Events::get_event_details($event_id);
        
        // If event not found, return error message
        if (!$event) {
            return '<p>' . __('Event not found.', 'quick-event-management') . '</p>';
        }
        
        // Check if registration is enabled
        if (!$event['registration_enabled']) {
            return '<p>' . __('Registration is not enabled for this event.', 'quick-event-management') . '</p>';
        }
        
        // Check if event is full
        if ($event['is_full']) {
            return '<p>' . __('This event is full.', 'quick-event-management') . '</p>';
        }
        
        // Start output buffering
        ob_start();
        
        // Include template
        include plugin_dir_path(__FILE__) . 'partials/quick-event-management-registration-form.php';
        
        // Return the buffered content
        return ob_get_clean();
    }

    /**
     * Shortcode for displaying event calendar.
     *
     * @since    1.0.0
     * @param    array    $atts    Shortcode attributes.
     * @return   string   Shortcode output.
     */
    public function calendar_shortcode($atts) {
        $atts = shortcode_atts(array(
            'category' => '',
            'tag' => '',
            'months' => 1,
        ), $atts, 'qem_calendar');
        
        // Build query args
        $args = array(
            'posts_per_page' => -1,
        );
        
        // Add category filter
        if (!empty($atts['category'])) {
            $args['tax_query'][] = array(
                'taxonomy' => 'qem_event_category',
                'field' => 'slug',
                'terms' => explode(',', $atts['category']),
            );
        }
        
        // Add tag filter
        if (!empty($atts['tag'])) {
            $args['tax_query'][] = array(
                'taxonomy' => 'qem_event_tag',
                'field' => 'slug',
                'terms' => explode(',', $atts['tag']),
            );
        }
        
        // Get events
        $events = Quick_Event_Management_Events::get_events($args);
        
        // Start output buffering
        ob_start();
        
        // Include template
        include plugin_dir_path(__FILE__) . 'partials/quick-event-management-calendar.php';
        
        // Return the buffered content
        return ob_get_clean();
    }

    /**
     * Process event registration.
     *
     * @since    1.0.0
     */
    public function process_event_registration() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qem_ajax_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'quick-event-management')));
        }
        
        // Get form data
        $event_id = isset($_POST['event_id']) ? intval($_POST['event_id']) : 0;
        $name = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
        $email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
        $phone = isset($_POST['phone']) ? sanitize_text_field($_POST['phone']) : '';
        $tickets = isset($_POST['tickets']) ? absint($_POST['tickets']) : 1;
        $notes = isset($_POST['notes']) ? sanitize_textarea_field($_POST['notes']) : '';
        
        // Validate required fields
        if (!$event_id || !$name || !$email) {
            wp_send_json_error(array('message' => __('Please fill in all required fields.', 'quick-event-management')));
        }
        
        // Register attendee
        $registration_data = array(
            'event_id' => $event_id,
            'name' => $name,
            'email' => $email,
            'phone' => $phone,
            'tickets' => $tickets,
            'notes' => $notes,
        );
        
        $result = Quick_Event_Management_Registrations::register_attendee($registration_data);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        } else {
            wp_send_json_success(array(
                'message' => __('Registration successful!', 'quick-event-management'),
                'registration_id' => $result,
            ));
        }
    }
}
