<?php

/**
 * Handles event registration functionality.
 *
 * @since      1.0.0
 */
class Quick_Event_Management_Registrations {

    /**
     * Register an attendee for an event.
     *
     * @since    1.0.0
     * @param    array    $data    Registration data.
     * @return   int|WP_Error    Registration ID on success, WP_Error on failure.
     */
    public static function register_attendee($data) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qem_registrations';
        
        // Validate required fields
        if (empty($data['event_id']) || empty($data['name']) || empty($data['email'])) {
            return new WP_Error('missing_required_fields', __('Missing required fields', 'quick-event-management'));
        }
        
        // Check if event exists
        $event = get_post($data['event_id']);
        if (!$event || $event->post_type !== 'qem_event') {
            return new WP_Error('invalid_event', __('Invalid event', 'quick-event-management'));
        }
        
        // Check if registration is enabled for this event
        $registration_enabled = get_post_meta($data['event_id'], '_qem_event_registration_enabled', true);
        if ($registration_enabled !== 'yes') {
            return new WP_Error('registration_disabled', __('Registration is not enabled for this event', 'quick-event-management'));
        }
        
        // Check if event is full
        if (Quick_Event_Management_Events::is_event_full($data['event_id'])) {
            return new WP_Error('event_full', __('This event is full', 'quick-event-management'));
        }
        
        // Set default values
        $tickets = isset($data['tickets']) ? absint($data['tickets']) : 1;
        $status = 'confirmed';
        
        // Check if approval is required
        $settings = get_option('qem_settings', array());
        if (!empty($settings['require_approval']) && $settings['require_approval']) {
            $status = 'pending';
        }
        
        // Check available spaces
        $available_spaces = Quick_Event_Management_Events::get_available_spaces($data['event_id']);
        if ($available_spaces !== null && $tickets > $available_spaces) {
            return new WP_Error('not_enough_spaces', __('Not enough spaces available', 'quick-event-management'));
        }
        
        // Insert registration
        $result = $wpdb->insert(
            $table_name,
            array(
                'event_id' => $data['event_id'],
                'name' => sanitize_text_field($data['name']),
                'email' => sanitize_email($data['email']),
                'phone' => isset($data['phone']) ? sanitize_text_field($data['phone']) : '',
                'tickets' => $tickets,
                'status' => $status,
                'registration_date' => current_time('mysql'),
                'notes' => isset($data['notes']) ? sanitize_textarea_field($data['notes']) : '',
            ),
            array('%d', '%s', '%s', '%s', '%d', '%s', '%s', '%s')
        );
        
        if ($result === false) {
            return new WP_Error('db_error', __('Failed to save registration', 'quick-event-management'));
        }
        
        $registration_id = $wpdb->insert_id;
        
        // Send notifications
        self::send_notifications($registration_id);
        
        return $registration_id;
    }

    /**
     * Get registrations for an event.
     *
     * @since    1.0.0
     * @param    int      $event_id    Event post ID.
     * @param    array    $args        Query arguments.
     * @return   array    Array of registrations.
     */
    public static function get_registrations($event_id, $args = array()) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qem_registrations';
        
        $defaults = array(
            'status' => 'confirmed',
            'orderby' => 'registration_date',
            'order' => 'ASC',
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $query = $wpdb->prepare(
            "SELECT * FROM $table_name WHERE event_id = %d",
            $event_id
        );
        
        if ($args['status'] !== 'all') {
            $query .= $wpdb->prepare(" AND status = %s", $args['status']);
        }
        
        $query .= " ORDER BY {$args['orderby']} {$args['order']}";
        
        return $wpdb->get_results($query);
    }

    /**
     * Update registration status.
     *
     * @since    1.0.0
     * @param    int      $registration_id    Registration ID.
     * @param    string   $status             New status.
     * @return   bool     True on success, false on failure.
     */
    public static function update_status($registration_id, $status) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qem_registrations';
        
        $result = $wpdb->update(
            $table_name,
            array('status' => $status),
            array('id' => $registration_id),
            array('%s'),
            array('%d')
        );
        
        return $result !== false;
    }

    /**
     * Delete a registration.
     *
     * @since    1.0.0
     * @param    int      $registration_id    Registration ID.
     * @return   bool     True on success, false on failure.
     */
    public static function delete_registration($registration_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qem_registrations';
        
        $result = $wpdb->delete(
            $table_name,
            array('id' => $registration_id),
            array('%d')
        );
        
        return $result !== false;
    }

    /**
     * Get a single registration.
     *
     * @since    1.0.0
     * @param    int      $registration_id    Registration ID.
     * @return   object|null    Registration object or null if not found.
     */
    public static function get_registration($registration_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qem_registrations';
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE id = %d",
            $registration_id
        ));
    }

    /**
     * Send notifications for a new registration.
     *
     * @since    1.0.0
     * @param    int      $registration_id    Registration ID.
     */
    public static function send_notifications($registration_id) {
        $registration = self::get_registration($registration_id);
        
        if (!$registration) {
            return;
        }
        
        $event = get_post($registration->event_id);
        $settings = get_option('qem_settings', array());
        
        // Admin notification
        if (!empty($settings['notify_admin']) && $settings['notify_admin']) {
            $admin_email = get_option('admin_email');
            $subject = !empty($settings['admin_email_subject']) ? $settings['admin_email_subject'] : __('New Event Registration', 'quick-event-management');
            
            $message = sprintf(
                __('New registration for event: %s', 'quick-event-management'),
                $event->post_title
            ) . "\n\n";
            
            $message .= sprintf(__('Name: %s', 'quick-event-management'), $registration->name) . "\n";
            $message .= sprintf(__('Email: %s', 'quick-event-management'), $registration->email) . "\n";
            
            if (!empty($registration->phone)) {
                $message .= sprintf(__('Phone: %s', 'quick-event-management'), $registration->phone) . "\n";
            }
            
            $message .= sprintf(__('Tickets: %d', 'quick-event-management'), $registration->tickets) . "\n";
            $message .= sprintf(__('Status: %s', 'quick-event-management'), $registration->status) . "\n";
            $message .= sprintf(__('Registration Date: %s', 'quick-event-management'), $registration->registration_date) . "\n";
            
            if (!empty($registration->notes)) {
                $message .= sprintf(__('Notes: %s', 'quick-event-management'), $registration->notes) . "\n";
            }
            
            wp_mail($admin_email, $subject, $message);
        }
        
        // Registrant notification
        if (!empty($settings['notify_registrant']) && $settings['notify_registrant']) {
            $subject = !empty($settings['registrant_email_subject']) ? $settings['registrant_email_subject'] : __('Your Event Registration Confirmation', 'quick-event-management');
            
            $message = sprintf(
                __('Thank you for registering for %s', 'quick-event-management'),
                $event->post_title
            ) . "\n\n";
            
            $event_details = Quick_Event_Management_Events::get_event_details($registration->event_id);
            
            $message .= sprintf(__('Event: %s', 'quick-event-management'), $event->post_title) . "\n";
            
            if (!empty($event_details['formatted_start_date'])) {
                $message .= sprintf(__('Date: %s', 'quick-event-management'), $event_details['formatted_start_date']) . "\n";
            }
            
            if (!empty($event_details['formatted_start_time'])) {
                $message .= sprintf(__('Time: %s', 'quick-event-management'), $event_details['formatted_start_time']) . "\n";
            }
            
            if (!empty($event_details['venue'])) {
                $message .= sprintf(__('Venue: %s', 'quick-event-management'), $event_details['venue']['name']) . "\n";
                
                if (!empty($event_details['venue']['address'])) {
                    $message .= sprintf(__('Address: %s', 'quick-event-management'), $event_details['venue']['address']) . "\n";
                }
            }
            
            $message .= "\n" . sprintf(__('Tickets: %d', 'quick-event-management'), $registration->tickets) . "\n";
            
            if ($registration->status === 'pending') {
                $message .= "\n" . __('Your registration is pending approval. We will notify you when it is confirmed.', 'quick-event-management') . "\n";
            }
            
            wp_mail($registration->email, $subject, $message);
        }
    }
}
