<?php

/**
 * Handles event-specific functionality.
 *
 * @since      1.0.0
 */
class Quick_Event_Management_Events {

    /**
     * Get all events with optional filtering.
     *
     * @since    1.0.0
     * @param    array    $args    Query arguments.
     * @return   array    Array of event posts.
     */
    public static function get_events($args = array()) {
        $defaults = array(
            'post_type'      => 'qem_event',
            'posts_per_page' => -1,
            'post_status'    => 'publish',
            'orderby'        => 'meta_value',
            'meta_key'       => '_qem_event_start_date',
            'order'          => 'ASC',
        );

        $args = wp_parse_args($args, $defaults);
        
        // Filter by date if specified
        if (!empty($args['date_filter'])) {
            $meta_query = array();
            
            if ($args['date_filter'] === 'upcoming') {
                $meta_query[] = array(
                    'key'     => '_qem_event_start_date',
                    'value'   => date('Y-m-d'),
                    'compare' => '>=',
                    'type'    => 'DATE'
                );
            } elseif ($args['date_filter'] === 'past') {
                $meta_query[] = array(
                    'key'     => '_qem_event_start_date',
                    'value'   => date('Y-m-d'),
                    'compare' => '<',
                    'type'    => 'DATE'
                );
            }
            
            $args['meta_query'] = $meta_query;
        }
        
        return get_posts($args);
    }

    /**
     * Get upcoming events.
     *
     * @since    1.0.0
     * @param    int      $limit    Number of events to return.
     * @return   array    Array of event posts.
     */
    public static function get_upcoming_events($limit = 5) {
        return self::get_events(array(
            'posts_per_page' => $limit,
            'date_filter'    => 'upcoming',
        ));
    }

    /**
     * Get past events.
     *
     * @since    1.0.0
     * @param    int      $limit    Number of events to return.
     * @return   array    Array of event posts.
     */
    public static function get_past_events($limit = 5) {
        return self::get_events(array(
            'posts_per_page' => $limit,
            'date_filter'    => 'past',
            'order'          => 'DESC',
        ));
    }

    /**
     * Get event details.
     *
     * @since    1.0.0
     * @param    int      $event_id    Event post ID.
     * @return   array    Event details.
     */
    public static function get_event_details($event_id) {
        $event = get_post($event_id);
        
        if (!$event || $event->post_type !== 'qem_event') {
            return false;
        }
        
        $start_date = get_post_meta($event_id, '_qem_event_start_date', true);
        $end_date = get_post_meta($event_id, '_qem_event_end_date', true);
        $start_time = get_post_meta($event_id, '_qem_event_start_time', true);
        $end_time = get_post_meta($event_id, '_qem_event_end_time', true);
        $venue_id = get_post_meta($event_id, '_qem_event_venue', true);
        $max_attendees = get_post_meta($event_id, '_qem_event_max_attendees', true);
        $registration_enabled = get_post_meta($event_id, '_qem_event_registration_enabled', true);
        
        // Get venue details if a venue is set
        $venue = null;
        if ($venue_id) {
            $venue_post = get_post($venue_id);
            if ($venue_post) {
                $venue = array(
                    'id' => $venue_id,
                    'name' => $venue_post->post_title,
                    'description' => $venue_post->post_content,
                    'address' => get_post_meta($venue_id, '_qem_venue_address', true),
                    'city' => get_post_meta($venue_id, '_qem_venue_city', true),
                    'state' => get_post_meta($venue_id, '_qem_venue_state', true),
                    'zip' => get_post_meta($venue_id, '_qem_venue_zip', true),
                    'country' => get_post_meta($venue_id, '_qem_venue_country', true),
                );
            }
        }
        
        // Get current registration count
        $registration_count = self::get_registration_count($event_id);
        
        // Format dates and times according to settings
        $settings = get_option('qem_settings', array());
        $date_format = !empty($settings['date_format']) ? $settings['date_format'] : 'F j, Y';
        $time_format = !empty($settings['time_format']) ? $settings['time_format'] : 'g:i a';
        
        $formatted_start_date = $start_date ? date_i18n($date_format, strtotime($start_date)) : '';
        $formatted_end_date = $end_date ? date_i18n($date_format, strtotime($end_date)) : '';
        $formatted_start_time = $start_time ? date_i18n($time_format, strtotime($start_time)) : '';
        $formatted_end_time = $end_time ? date_i18n($time_format, strtotime($end_time)) : '';
        
        return array(
            'id' => $event_id,
            'title' => $event->post_title,
            'description' => $event->post_content,
            'excerpt' => $event->post_excerpt,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'start_time' => $start_time,
            'end_time' => $end_time,
            'formatted_start_date' => $formatted_start_date,
            'formatted_end_date' => $formatted_end_date,
            'formatted_start_time' => $formatted_start_time,
            'formatted_end_time' => $formatted_end_time,
            'venue' => $venue,
            'max_attendees' => $max_attendees,
            'registration_count' => $registration_count,
            'registration_enabled' => $registration_enabled === 'yes',
            'spaces_available' => $max_attendees ? ($max_attendees - $registration_count) : null,
            'is_full' => $max_attendees ? ($registration_count >= $max_attendees) : false,
            'permalink' => get_permalink($event_id),
            'thumbnail' => get_the_post_thumbnail_url($event_id, 'large'),
            'categories' => wp_get_post_terms($event_id, 'qem_event_category', array('fields' => 'all')),
            'tags' => wp_get_post_terms($event_id, 'qem_event_tag', array('fields' => 'all')),
        );
    }

    /**
     * Get registration count for an event.
     *
     * @since    1.0.0
     * @param    int      $event_id    Event post ID.
     * @return   int      Registration count.
     */
    public static function get_registration_count($event_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qem_registrations';
        
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT SUM(tickets) FROM $table_name WHERE event_id = %d AND status = 'confirmed'",
            $event_id
        ));
        
        return $count ? (int) $count : 0;
    }

    /**
     * Check if an event is full.
     *
     * @since    1.0.0
     * @param    int      $event_id    Event post ID.
     * @return   bool     True if event is full, false otherwise.
     */
    public static function is_event_full($event_id) {
        $max_attendees = get_post_meta($event_id, '_qem_event_max_attendees', true);
        
        if (!$max_attendees) {
            return false;
        }
        
        $registration_count = self::get_registration_count($event_id);
        
        return $registration_count >= $max_attendees;
    }

    /**
     * Get available spaces for an event.
     *
     * @since    1.0.0
     * @param    int      $event_id    Event post ID.
     * @return   int|null Number of available spaces, or null if unlimited.
     */
    public static function get_available_spaces($event_id) {
        $max_attendees = get_post_meta($event_id, '_qem_event_max_attendees', true);
        
        if (!$max_attendees) {
            return null;
        }
        
        $registration_count = self::get_registration_count($event_id);
        
        return max(0, $max_attendees - $registration_count);
    }
}
