<?php
/**
 * Simple script to check for BOM and whitespace issues in PHP files
 */

// Directory to scan
$dir = __DIR__;

// Function to check a file for BOM and whitespace issues
function check_file($file) {
    $content = file_get_contents($file);
    $issues = array();
    
    // Check for BOM
    if (substr($content, 0, 3) === "\xEF\xBB\xBF") {
        $issues[] = "Has BOM marker at the beginning";
    }
    
    // Check if file starts with <?php
    if (substr($content, 0, 5) !== "<?php") {
        $issues[] = "Doesn't start with <?php";
    }
    
    // Check for whitespace before <?php
    if (preg_match('/^\s+<\?php/', $content)) {
        $issues[] = "Has whitespace before <?php";
    }
    
    // Check for whitespace after closing PHP tag
    if (preg_match('/\?>\s+$/', $content)) {
        $issues[] = "Has whitespace after closing PHP tag";
    }
    
    // Check if file ends with closing PHP tag (not recommended for pure PHP files)
    if (substr(trim($content), -2) === "?>") {
        $issues[] = "Ends with closing PHP tag (not recommended for pure PHP files)";
    }
    
    return $issues;
}

// Get all PHP files recursively
$iterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dir));
$files = new RegexIterator($iterator, '/\.php$/i', RecursiveRegexIterator::GET_MATCH);

// Check each file
$problem_files = array();
foreach ($files as $file) {
    $file_path = $file[0];
    
    // Skip this checker script
    if (basename($file_path) === 'check_files.php') {
        continue;
    }
    
    $issues = check_file($file_path);
    if (!empty($issues)) {
        $problem_files[$file_path] = $issues;
    }
}

// Output results
if (empty($problem_files)) {
    echo "No issues found in PHP files.\n";
} else {
    echo "Found issues in the following files:\n\n";
    foreach ($problem_files as $file => $issues) {
        echo "File: " . basename($file) . "\n";
        echo "Path: " . $file . "\n";
        echo "Issues:\n";
        foreach ($issues as $issue) {
            echo "- " . $issue . "\n";
        }
        echo "\n";
    }
}
