<?php
/**
 * Admin view for displaying all registrations
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Quick_Event_Management
 */

// Security check
if (!defined('ABSPATH')) {
    exit;
}

// Get registrations from database
global $wpdb;
$table_name = $wpdb->prefix . 'qem_registrations';

// Check if table exists
$table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") === $table_name;

// Setup filters
$event_filter = isset($_GET['event_id']) ? intval($_GET['event_id']) : 0;
$status_filter = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';

// Build query
$sql = "SELECT r.*, p.post_title as event_title 
        FROM {$table_name} r
        LEFT JOIN {$wpdb->posts} p ON r.event_id = p.ID
        WHERE 1=1";

$sql_params = array();

if ($event_filter > 0) {
    $sql .= " AND r.event_id = %d";
    $sql_params[] = $event_filter;
}

if (!empty($status_filter)) {
    $sql .= " AND r.status = %s";
    $sql_params[] = $status_filter;
}

$sql .= " ORDER BY r.registration_date DESC";

// Prepare the query if we have parameters
if (!empty($sql_params)) {
    $sql = $wpdb->prepare($sql, $sql_params);
}

// Get registrations
$registrations = $table_exists ? $wpdb->get_results($sql) : array();

// Get all events for the filter dropdown
$events = get_posts(array(
    'post_type' => 'qem_event',
    'posts_per_page' => -1,
    'orderby' => 'title',
    'order' => 'ASC',
));

// Status options
$status_options = array(
    'pending' => __('Pending', 'quick-event-management'),
    'confirmed' => __('Confirmed', 'quick-event-management'),
    'cancelled' => __('Cancelled', 'quick-event-management'),
);
?>

<div class="wrap">
    <h1><?php echo esc_html__('Event Registrations', 'quick-event-management'); ?></h1>
    
    <?php if (!$table_exists): ?>
        <div class="notice notice-error">
            <p><?php _e('The registrations table does not exist. Please deactivate and reactivate the plugin to create it.', 'quick-event-management'); ?></p>
        </div>
    <?php else: ?>
    
    <div class="qem-admin-filters">
        <form method="get">
            <input type="hidden" name="post_type" value="qem_event">
            <input type="hidden" name="page" value="qem-registrations">
            
            <select name="event_id">
                <option value=""><?php _e('All Events', 'quick-event-management'); ?></option>
                <?php foreach ($events as $event): ?>
                    <option value="<?php echo esc_attr($event->ID); ?>" <?php selected($event_filter, $event->ID); ?>>
                        <?php echo esc_html($event->post_title); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            
            <select name="status">
                <option value=""><?php _e('All Statuses', 'quick-event-management'); ?></option>
                <?php foreach ($status_options as $value => $label): ?>
                    <option value="<?php echo esc_attr($value); ?>" <?php selected($status_filter, $value); ?>>
                        <?php echo esc_html($label); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            
            <input type="submit" class="button" value="<?php _e('Filter', 'quick-event-management'); ?>">
        </form>
    </div>
    
    <?php if (empty($registrations)): ?>
        <div class="qem-no-registrations">
            <p><?php _e('No registrations found.', 'quick-event-management'); ?></p>
        </div>
    <?php else: ?>
        <div class="qem-admin-actions">
            <button id="qem-export-csv" class="button">
                <?php _e('Export to CSV', 'quick-event-management'); ?>
            </button>
        </div>
        
        <table class="wp-list-table widefat fixed striped qem-registrations-table">
            <thead>
                <tr>
                    <th><?php _e('ID', 'quick-event-management'); ?></th>
                    <th><?php _e('Event', 'quick-event-management'); ?></th>
                    <th><?php _e('Name', 'quick-event-management'); ?></th>
                    <th><?php _e('Email', 'quick-event-management'); ?></th>
                    <th><?php _e('Phone', 'quick-event-management'); ?></th>
                    <th><?php _e('Tickets', 'quick-event-management'); ?></th>
                    <th><?php _e('Status', 'quick-event-management'); ?></th>
                    <th><?php _e('Registration Date', 'quick-event-management'); ?></th>
                    <th><?php _e('Actions', 'quick-event-management'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($registrations as $registration): ?>
                    <tr>
                        <td><?php echo esc_html($registration->id); ?></td>
                        <td>
                            <a href="<?php echo esc_url(get_edit_post_link($registration->event_id)); ?>">
                                <?php echo esc_html($registration->event_title); ?>
                            </a>
                        </td>
                        <td><?php echo esc_html($registration->name); ?></td>
                        <td><?php echo esc_html($registration->email); ?></td>
                        <td><?php echo esc_html($registration->phone); ?></td>
                        <td><?php echo esc_html($registration->tickets); ?></td>
                        <td>
                            <span class="qem-status qem-status-<?php echo esc_attr($registration->status); ?>">
                                <?php echo esc_html($status_options[$registration->status] ?? $registration->status); ?>
                            </span>
                        </td>
                        <td><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($registration->registration_date))); ?></td>
                        <td>
                            <div class="qem-actions">
                                <?php if ($registration->status !== 'confirmed'): ?>
                                    <a href="#" class="qem-action-confirm" data-id="<?php echo esc_attr($registration->id); ?>" data-nonce="<?php echo wp_create_nonce('qem_confirm_registration'); ?>">
                                        <?php _e('Confirm', 'quick-event-management'); ?>
                                    </a>
                                <?php endif; ?>
                                
                                <?php if ($registration->status !== 'cancelled'): ?>
                                    <a href="#" class="qem-action-cancel" data-id="<?php echo esc_attr($registration->id); ?>" data-nonce="<?php echo wp_create_nonce('qem_cancel_registration'); ?>">
                                        <?php _e('Cancel', 'quick-event-management'); ?>
                                    </a>
                                <?php endif; ?>
                                
                                <a href="#" class="qem-action-delete" data-id="<?php echo esc_attr($registration->id); ?>" data-nonce="<?php echo wp_create_nonce('qem_delete_registration'); ?>">
                                    <?php _e('Delete', 'quick-event-management'); ?>
                                </a>
                                
                                <a href="#" class="qem-action-view-notes" data-id="<?php echo esc_attr($registration->id); ?>">
                                    <?php _e('View Notes', 'quick-event-management'); ?>
                                </a>
                            </div>
                            
                            <div id="qem-notes-<?php echo esc_attr($registration->id); ?>" class="qem-notes-popup" style="display: none;">
                                <div class="qem-notes-content">
                                    <h3><?php _e('Registration Notes', 'quick-event-management'); ?></h3>
                                    <div class="qem-notes-text">
                                        <?php echo !empty($registration->notes) ? nl2br(esc_html($registration->notes)) : __('No notes available.', 'quick-event-management'); ?>
                                    </div>
                                    <button class="button qem-close-notes"><?php _e('Close', 'quick-event-management'); ?></button>
                                </div>
                            </div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
    <?php endif; ?>
</div>

<script>
jQuery(document).ready(function($) {
    // Status change actions
    $('.qem-action-confirm, .qem-action-cancel').on('click', function(e) {
        e.preventDefault();
        
        var $link = $(this);
        var registrationId = $link.data('id');
        var nonce = $link.data('nonce');
        var newStatus = $link.hasClass('qem-action-confirm') ? 'confirmed' : 'cancelled';
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qem_update_registration_status',
                registration_id: registrationId,
                status: newStatus,
                nonce: nonce
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data.message || 'Error updating registration status.');
                }
            },
            error: function() {
                alert('Server error. Please try again.');
            }
        });
    });
    
    // Delete action
    $('.qem-action-delete').on('click', function(e) {
        e.preventDefault();
        
        if (!confirm('<?php _e('Are you sure you want to delete this registration? This action cannot be undone.', 'quick-event-management'); ?>')) {
            return;
        }
        
        var $link = $(this);
        var registrationId = $link.data('id');
        var nonce = $link.data('nonce');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'qem_delete_registration',
                registration_id: registrationId,
                nonce: nonce
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data.message || 'Error deleting registration.');
                }
            },
            error: function() {
                alert('Server error. Please try again.');
            }
        });
    });
    
    // View notes popup
    $('.qem-action-view-notes').on('click', function(e) {
        e.preventDefault();
        var registrationId = $(this).data('id');
        $('#qem-notes-' + registrationId).fadeIn(200);
    });
    
    // Close notes popup
    $('.qem-close-notes').on('click', function() {
        $(this).closest('.qem-notes-popup').fadeOut(200);
    });
    
    // Export to CSV
    $('#qem-export-csv').on('click', function() {
        var eventId = $('select[name="event_id"]').val() || '';
        var status = $('select[name="status"]').val() || '';
        
        var url = ajaxurl + '?action=qem_export_registrations';
        if (eventId) {
            url += '&event_id=' + eventId;
        }
        if (status) {
            url += '&status=' + status;
        }
        url += '&nonce=' + '<?php echo wp_create_nonce('qem_export_registrations'); ?>';
        
        window.location.href = url;
    });
});
</script>

<style>
.qem-admin-filters {
    margin: 20px 0;
    display: flex;
    align-items: center;
}

.qem-admin-filters select {
    margin-right: 10px;
}

.qem-admin-actions {
    margin: 20px 0;
    text-align: right;
}

.qem-status {
    display: inline-block;
    padding: 3px 8px;
    border-radius: 3px;
    font-size: 12px;
    font-weight: bold;
    text-transform: uppercase;
}

.qem-status-pending {
    background-color: #f8dda7;
    color: #94660c;
}

.qem-status-confirmed {
    background-color: #c6e1c6;
    color: #5b841b;
}

.qem-status-cancelled {
    background-color: #f1adad;
    color: #a00;
}

.qem-actions {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
}

.qem-actions a {
    margin-right: 8px;
    text-decoration: none;
}

.qem-notes-popup {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background-color: rgba(0, 0, 0, 0.5);
    z-index: 9999;
    display: flex;
    align-items: center;
    justify-content: center;
}

.qem-notes-content {
    background-color: #fff;
    padding: 20px;
    border-radius: 5px;
    max-width: 600px;
    width: 100%;
    max-height: 80vh;
    overflow-y: auto;
}

.qem-notes-text {
    margin: 15px 0;
    padding: 10px;
    background-color: #f9f9f9;
    border: 1px solid #e5e5e5;
    border-radius: 3px;
    min-height: 100px;
}

.qem-no-registrations {
    padding: 20px;
    background-color: #f9f9f9;
    border: 1px solid #e5e5e5;
    border-radius: 3px;
    text-align: center;
    margin: 20px 0;
}
</style>
