<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @since      1.0.0
 */
class Quick_Event_Management_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        
        // Register AJAX handlers
        add_action('wp_ajax_qem_update_registration_status', array($this, 'ajax_update_registration_status'));
        add_action('wp_ajax_qem_delete_registration', array($this, 'ajax_delete_registration'));
        add_action('wp_ajax_qem_export_registrations', array($this, 'ajax_export_registrations'));
        add_action('wp_ajax_qem_add_venue', array($this, 'ajax_add_venue'));
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/quick-event-management-admin.css', array(), $this->version, 'all');
        
        // Add jQuery UI datepicker styles
        wp_enqueue_style('jquery-ui', 'https://code.jquery.com/ui/1.12.1/themes/base/jquery-ui.css');
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/quick-event-management-admin.js', array('jquery', 'jquery-ui-datepicker', 'jquery-ui-timepicker'), $this->version, false);
        
        // Add jQuery UI timepicker addon
        wp_enqueue_script('jquery-ui-timepicker', plugin_dir_url(__FILE__) . 'js/jquery-ui-timepicker-addon.min.js', array('jquery', 'jquery-ui-datepicker'), '1.6.3', false);
    }

    /**
     * Add plugin admin menu.
     *
     * @since    1.0.0
     */
    public function add_plugin_admin_menu() {
        // Main menu item is already added by the post type registration
        
        // Add submenu pages
        add_submenu_page(
            'edit.php?post_type=qem_event',
            __('Registrations', 'quick-event-management'),
            __('Registrations', 'quick-event-management'),
            'manage_options',
            'qem-registrations',
            array($this, 'display_registrations_page')
        );
        
        add_submenu_page(
            'edit.php?post_type=qem_event',
            __('Settings', 'quick-event-management'),
            __('Settings', 'quick-event-management'),
            'manage_options',
            'qem-settings',
            array($this, 'display_settings_page')
        );
    }

    /**
     * Display registrations page.
     *
     * @since    1.0.0
     */
    public function display_registrations_page() {
        // Get event ID from query string
        $event_id = isset($_GET['event_id']) ? intval($_GET['event_id']) : 0;
        
        // Include the registrations page template
        include_once plugin_dir_path(__FILE__) . 'partials/quick-event-management-admin-registrations.php';
    }

    /**
     * Display settings page.
     *
     * @since    1.0.0
     */
    public function display_settings_page() {
        // Process form submission
        if (isset($_POST['qem_settings_submit']) && check_admin_referer('qem_settings_nonce', 'qem_settings_nonce')) {
            $settings = array(
                'event_slug' => sanitize_text_field($_POST['event_slug']),
                'events_per_page' => absint($_POST['events_per_page']),
                'date_format' => sanitize_text_field($_POST['date_format']),
                'time_format' => sanitize_text_field($_POST['time_format']),
                'require_approval' => isset($_POST['require_approval']) ? true : false,
                'notify_admin' => isset($_POST['notify_admin']) ? true : false,
                'notify_registrant' => isset($_POST['notify_registrant']) ? true : false,
                'admin_email_subject' => sanitize_text_field($_POST['admin_email_subject']),
                'registrant_email_subject' => sanitize_text_field($_POST['registrant_email_subject']),
            );
            
            update_option('qem_settings', $settings);
            
            // Flush rewrite rules if slug changed
            flush_rewrite_rules();
            
            add_settings_error('qem_settings', 'settings_updated', __('Settings saved.', 'quick-event-management'), 'updated');
        }
        
        // Include the settings page template
        include_once plugin_dir_path(__FILE__) . 'partials/quick-event-management-admin-settings.php';
    }

    /**
     * Add metaboxes for event details.
     *
     * @since    1.0.0
     */
    public function add_event_metaboxes() {
        add_meta_box(
            'qem_event_details',
            __('Event Details', 'quick-event-management'),
            array($this, 'render_event_details_metabox'),
            'qem_event',
            'normal',
            'high'
        );
        
        add_meta_box(
            'qem_event_venue',
            __('Event Venue', 'quick-event-management'),
            array($this, 'render_event_venue_metabox'),
            'qem_event',
            'normal',
            'default'
        );
        
        add_meta_box(
            'qem_event_registration',
            __('Registration Settings', 'quick-event-management'),
            array($this, 'render_event_registration_metabox'),
            'qem_event',
            'normal',
            'default'
        );
        
        add_meta_box(
            'qem_event_registrations',
            __('Registrations', 'quick-event-management'),
            array($this, 'render_event_registrations_metabox'),
            'qem_event',
            'side',
            'default'
        );
    }

    /**
     * Render event details metabox.
     *
     * @since    1.0.0
     * @param    WP_Post    $post    The post object.
     */
    public function render_event_details_metabox($post) {
        // Add nonce for security
        wp_nonce_field('qem_event_details_nonce', 'qem_event_details_nonce');
        
        // Get saved values
        $start_date = get_post_meta($post->ID, '_qem_event_start_date', true);
        $end_date = get_post_meta($post->ID, '_qem_event_end_date', true);
        $start_time = get_post_meta($post->ID, '_qem_event_start_time', true);
        $end_time = get_post_meta($post->ID, '_qem_event_end_time', true);
        
        // Include the metabox template
        include plugin_dir_path(__FILE__) . 'partials/quick-event-management-admin-event-details.php';
    }

    /**
     * Render event venue metabox.
     *
     * @since    1.0.0
     * @param    WP_Post    $post    The post object.
     */
    public function render_event_venue_metabox($post) {
        // Get saved values
        $venue_id = get_post_meta($post->ID, '_qem_event_venue', true);
        
        // Get all venues
        $venues = get_posts(array(
            'post_type' => 'qem_venue',
            'posts_per_page' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ));
        
        // Include the metabox template
        include plugin_dir_path(__FILE__) . 'partials/quick-event-management-admin-event-venue.php';
    }

    /**
     * Render event registration metabox.
     *
     * @since    1.0.0
     * @param    WP_Post    $post    The post object.
     */
    public function render_event_registration_metabox($post) {
        // Get saved values
        $registration_enabled = get_post_meta($post->ID, '_qem_event_registration_enabled', true);
        $max_attendees = get_post_meta($post->ID, '_qem_event_max_attendees', true);
        
        // Include the metabox template
        include plugin_dir_path(__FILE__) . 'partials/quick-event-management-admin-event-registration.php';
    }

    /**
     * Render event registrations metabox.
     *
     * @since    1.0.0
     * @param    WP_Post    $post    The post object.
     */
    public function render_event_registrations_metabox($post) {
        // Get registration count
        $registration_count = Quick_Event_Management_Events::get_registration_count($post->ID);
        $max_attendees = get_post_meta($post->ID, '_qem_event_max_attendees', true);
        
        // Include the metabox template
        include plugin_dir_path(__FILE__) . 'partials/quick-event-management-admin-event-registrations.php';
    }

    /**
     * Save event metaboxes.
     *
     * @since    1.0.0
     * @param    int       $post_id    The post ID.
     * @param    WP_Post   $post       The post object.
     */
    public function save_event_metaboxes($post_id, $post) {
        // Check if our nonce is set
        if (!isset($_POST['qem_event_details_nonce'])) {
            return;
        }
        
        // Verify that the nonce is valid
        if (!wp_verify_nonce($_POST['qem_event_details_nonce'], 'qem_event_details_nonce')) {
            return;
        }
        
        // If this is an autosave, our form has not been submitted, so we don't want to do anything
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Check the user's permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Save event details
        if (isset($_POST['qem_event_start_date'])) {
            update_post_meta($post_id, '_qem_event_start_date', sanitize_text_field($_POST['qem_event_start_date']));
        }
        
        if (isset($_POST['qem_event_end_date'])) {
            update_post_meta($post_id, '_qem_event_end_date', sanitize_text_field($_POST['qem_event_end_date']));
        }
        
        if (isset($_POST['qem_event_start_time'])) {
            update_post_meta($post_id, '_qem_event_start_time', sanitize_text_field($_POST['qem_event_start_time']));
        }
        
        if (isset($_POST['qem_event_end_time'])) {
            update_post_meta($post_id, '_qem_event_end_time', sanitize_text_field($_POST['qem_event_end_time']));
        }
        
        // Save venue
        if (isset($_POST['qem_event_venue'])) {
            update_post_meta($post_id, '_qem_event_venue', absint($_POST['qem_event_venue']));
        }
        
        // Save registration settings
        $registration_enabled = isset($_POST['qem_event_registration_enabled']) ? 'yes' : 'no';
        update_post_meta($post_id, '_qem_event_registration_enabled', $registration_enabled);
        
        if (isset($_POST['qem_event_max_attendees'])) {
            update_post_meta($post_id, '_qem_event_max_attendees', absint($_POST['qem_event_max_attendees']));
        }
    }
    
    /**
     * AJAX handler for updating registration status
     *
     * @since    1.0.0
     */
    public function ajax_update_registration_status() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qem_confirm_registration')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'quick-event-management')));
        }
        
        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'quick-event-management')));
        }
        
        // Get parameters
        $registration_id = isset($_POST['registration_id']) ? intval($_POST['registration_id']) : 0;
        $status = isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '';
        
        // Validate parameters
        if ($registration_id <= 0 || !in_array($status, array('pending', 'confirmed', 'cancelled'))) {
            wp_send_json_error(array('message' => __('Invalid parameters.', 'quick-event-management')));
        }
        
        // Update registration status
        global $wpdb;
        $table_name = $wpdb->prefix . 'qem_registrations';
        
        $result = $wpdb->update(
            $table_name,
            array('status' => $status),
            array('id' => $registration_id),
            array('%s'),
            array('%d')
        );
        
        if ($result === false) {
            wp_send_json_error(array('message' => __('Failed to update registration status.', 'quick-event-management')));
        }
        
        // Get registration details for notification
        $registration = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $registration_id));
        
        // Send notification email if status changed to confirmed
        if ($status === 'confirmed' && $registration) {
            $this->send_status_change_notification($registration);
        }
        
        wp_send_json_success(array('message' => __('Registration status updated successfully.', 'quick-event-management')));
    }
    
    /**
     * AJAX handler for deleting a registration
     *
     * @since    1.0.0
     */
    public function ajax_delete_registration() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qem_delete_registration')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'quick-event-management')));
        }
        
        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'quick-event-management')));
        }
        
        // Get parameters
        $registration_id = isset($_POST['registration_id']) ? intval($_POST['registration_id']) : 0;
        
        // Validate parameters
        if ($registration_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid registration ID.', 'quick-event-management')));
        }
        
        // Delete registration
        global $wpdb;
        $table_name = $wpdb->prefix . 'qem_registrations';
        
        $result = $wpdb->delete(
            $table_name,
            array('id' => $registration_id),
            array('%d')
        );
        
        if ($result === false) {
            wp_send_json_error(array('message' => __('Failed to delete registration.', 'quick-event-management')));
        }
        
        wp_send_json_success(array('message' => __('Registration deleted successfully.', 'quick-event-management')));
    }
    
    /**
     * AJAX handler for exporting registrations to CSV
     *
     * @since    1.0.0
     */
    public function ajax_export_registrations() {
        // Check nonce
        if (!isset($_GET['nonce']) || !wp_verify_nonce($_GET['nonce'], 'qem_export_registrations')) {
            wp_die(__('Security check failed.', 'quick-event-management'));
        }
        
        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have permission to perform this action.', 'quick-event-management'));
        }
        
        // Get parameters
        $event_id = isset($_GET['event_id']) ? intval($_GET['event_id']) : 0;
        $status = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
        
        // Build query
        global $wpdb;
        $table_name = $wpdb->prefix . 'qem_registrations';
        
        $sql = "SELECT r.*, p.post_title as event_title 
                FROM {$table_name} r
                LEFT JOIN {$wpdb->posts} p ON r.event_id = p.ID
                WHERE 1=1";
        
        $sql_params = array();
        
        if ($event_id > 0) {
            $sql .= " AND r.event_id = %d";
            $sql_params[] = $event_id;
        }
        
        if (!empty($status)) {
            $sql .= " AND r.status = %s";
            $sql_params[] = $status;
        }
        
        $sql .= " ORDER BY r.registration_date DESC";
        
        // Prepare the query if we have parameters
        if (!empty($sql_params)) {
            $sql = $wpdb->prepare($sql, $sql_params);
        }
        
        // Get registrations
        $registrations = $wpdb->get_results($sql);
        
        // Set up CSV headers
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=event-registrations-' . date('Y-m-d') . '.csv');
        
        // Create a file pointer connected to the output stream
        $output = fopen('php://output', 'w');
        
        // Set column headers
        fputcsv($output, array(
            __('ID', 'quick-event-management'),
            __('Event', 'quick-event-management'),
            __('Name', 'quick-event-management'),
            __('Email', 'quick-event-management'),
            __('Phone', 'quick-event-management'),
            __('Tickets', 'quick-event-management'),
            __('Status', 'quick-event-management'),
            __('Registration Date', 'quick-event-management'),
            __('Notes', 'quick-event-management')
        ));
        
        // Output each row of data
        foreach ($registrations as $registration) {
            fputcsv($output, array(
                $registration->id,
                $registration->event_title,
                $registration->name,
                $registration->email,
                $registration->phone,
                $registration->tickets,
                $registration->status,
                $registration->registration_date,
                $registration->notes
            ));
        }
        
        exit;
    }
    
    /**
     * AJAX handler for adding a new venue
     *
     * @since    1.0.0
     */
    public function ajax_add_venue() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qem_add_venue_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'quick-event-management')));
        }
        
        // Check capabilities
        if (!current_user_can('publish_posts')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'quick-event-management')));
        }
        
        // Get venue data
        $venue_name = isset($_POST['venue_name']) ? sanitize_text_field($_POST['venue_name']) : '';
        $venue_address = isset($_POST['venue_address']) ? sanitize_text_field($_POST['venue_address']) : '';
        $venue_city = isset($_POST['venue_city']) ? sanitize_text_field($_POST['venue_city']) : '';
        $venue_state = isset($_POST['venue_state']) ? sanitize_text_field($_POST['venue_state']) : '';
        $venue_zip = isset($_POST['venue_zip']) ? sanitize_text_field($_POST['venue_zip']) : '';
        $venue_country = isset($_POST['venue_country']) ? sanitize_text_field($_POST['venue_country']) : '';
        
        // Validate venue name
        if (empty($venue_name)) {
            wp_send_json_error(array('message' => __('Venue name is required.', 'quick-event-management')));
        }
        
        // Create venue post
        $venue_data = array(
            'post_title'   => $venue_name,
            'post_content' => '',
            'post_status'  => 'publish',
            'post_type'    => 'qem_venue',
        );
        
        $venue_id = wp_insert_post($venue_data);
        
        if (is_wp_error($venue_id)) {
            wp_send_json_error(array('message' => $venue_id->get_error_message()));
        }
        
        // Save venue meta
        update_post_meta($venue_id, '_qem_venue_address', $venue_address);
        update_post_meta($venue_id, '_qem_venue_city', $venue_city);
        update_post_meta($venue_id, '_qem_venue_state', $venue_state);
        update_post_meta($venue_id, '_qem_venue_zip', $venue_zip);
        update_post_meta($venue_id, '_qem_venue_country', $venue_country);
        
        // Return success with venue data
        wp_send_json_success(array(
            'message' => __('Venue added successfully.', 'quick-event-management'),
            'venue' => array(
                'id' => $venue_id,
                'name' => $venue_name
            )
        ));
    }
    
    /**
     * Send notification email when registration status changes
     *
     * @since    1.0.0
     * @param    object    $registration    Registration object
     */
    private function send_status_change_notification($registration) {
        // Get event details
        $event = get_post($registration->event_id);
        if (!$event) {
            return;
        }
        
        // Get plugin settings
        $settings = get_option('qem_settings', array());
        $notify_registrant = isset($settings['notify_registrant']) ? $settings['notify_registrant'] : true;
        
        if (!$notify_registrant) {
            return;
        }
        
        // Get email subject
        $subject = isset($settings['registrant_email_subject']) 
            ? $settings['registrant_email_subject'] 
            : __('Your Event Registration Status', 'quick-event-management');
        
        // Replace placeholders in subject
        $subject = str_replace(
            array('[event_title]'),
            array($event->post_title),
            $subject
        );
        
        // Build email message
        $message = sprintf(
            __('Hello %s,', 'quick-event-management') . "\n\n" .
            __('Your registration for "%s" has been confirmed.', 'quick-event-management') . "\n\n" .
            __('Event Details:', 'quick-event-management') . "\n" .
            __('Date: %s', 'quick-event-management') . "\n" .
            __('Tickets: %d', 'quick-event-management') . "\n\n" .
            __('Thank you for your registration!', 'quick-event-management'),
            $registration->name,
            $event->post_title,
            get_post_meta($event->ID, '_qem_event_start_date', true),
            $registration->tickets
        );
        
        // Send email
        wp_mail($registration->email, $subject, $message);
    }
}
