<?php
/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * @since      1.0.0
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 */

/**
 * The core plugin class.
 *
 * @since      1.0.0
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Push_Notifications {

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Quick_Push_Notifications_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if ( defined( 'QUICK_PUSH_NOTIFICATIONS_VERSION' ) ) {
            $this->version = QUICK_PUSH_NOTIFICATIONS_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'quick-push-notifications';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - Quick_Push_Notifications_Loader. Orchestrates the hooks of the plugin.
     * - Quick_Push_Notifications_i18n. Defines internationalization functionality.
     * - Quick_Push_Notifications_Admin. Defines all hooks for the admin area.
     * - Quick_Push_Notifications_Public. Defines all hooks for the public side of the site.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once QUICK_PUSH_NOTIFICATIONS_PLUGIN_DIR . 'includes/class-quick-push-notifications-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once QUICK_PUSH_NOTIFICATIONS_PLUGIN_DIR . 'includes/class-quick-push-notifications-i18n.php';

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once QUICK_PUSH_NOTIFICATIONS_PLUGIN_DIR . 'admin/class-quick-push-notifications-admin.php';

        /**
         * The class responsible for defining all actions that occur in the public-facing
         * side of the site.
         */
        require_once QUICK_PUSH_NOTIFICATIONS_PLUGIN_DIR . 'public/class-quick-push-notifications-public.php';

        /**
         * The class responsible for handling subscribers.
         */
        require_once QUICK_PUSH_NOTIFICATIONS_PLUGIN_DIR . 'includes/class-quick-push-notifications-subscriber.php';

        /**
         * The class responsible for handling notifications.
         */
        require_once QUICK_PUSH_NOTIFICATIONS_PLUGIN_DIR . 'includes/class-quick-push-notifications-notification.php';

        /**
         * The class responsible for handling Firebase integration.
         */
        require_once QUICK_PUSH_NOTIFICATIONS_PLUGIN_DIR . 'includes/class-quick-push-notifications-firebase.php';

        /**
         * The class responsible for handling API endpoints.
         */
        require_once QUICK_PUSH_NOTIFICATIONS_PLUGIN_DIR . 'includes/class-quick-push-notifications-api.php';

        $this->loader = new Quick_Push_Notifications_Loader();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Quick_Push_Notifications_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {

        $plugin_i18n = new Quick_Push_Notifications_i18n();

        $this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {

        $plugin_admin = new Quick_Push_Notifications_Admin( $this->get_plugin_name(), $this->get_version() );

        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
        
        // Add menu items
        $this->loader->add_action( 'admin_menu', $plugin_admin, 'add_plugin_admin_menu' );
        
        // Add Settings link to the plugin
        $this->loader->add_filter( 'plugin_action_links_' . QUICK_PUSH_NOTIFICATIONS_PLUGIN_BASENAME, $plugin_admin, 'add_action_links' );
        
        // Register settings
        $this->loader->add_action( 'admin_init', $plugin_admin, 'register_settings' );
        
        // AJAX handlers
        $this->loader->add_action( 'wp_ajax_qpn_send_test_notification', $plugin_admin, 'ajax_send_test_notification' );
        $this->loader->add_action( 'wp_ajax_qpn_get_subscribers', $plugin_admin, 'ajax_get_subscribers' );
        $this->loader->add_action( 'wp_ajax_qpn_delete_subscriber', $plugin_admin, 'ajax_delete_subscriber' );
        $this->loader->add_action( 'wp_ajax_qpn_get_notifications', $plugin_admin, 'ajax_get_notifications' );
        $this->loader->add_action( 'wp_ajax_qpn_save_notification', $plugin_admin, 'ajax_save_notification' );
        $this->loader->add_action( 'wp_ajax_qpn_send_notification', $plugin_admin, 'ajax_send_notification' );
        $this->loader->add_action( 'wp_ajax_qpn_delete_notification', $plugin_admin, 'ajax_delete_notification' );
        
        // Auto-push for new posts
        $this->loader->add_action( 'transition_post_status', $plugin_admin, 'check_post_status_for_notification', 10, 3 );
        
        // Scheduled notifications
        $this->loader->add_action( 'qpn_send_scheduled_notifications', $plugin_admin, 'send_scheduled_notifications' );
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks() {

        $plugin_public = new Quick_Push_Notifications_Public( $this->get_plugin_name(), $this->get_version() );

        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
        
        // Add service worker
        $this->loader->add_action( 'wp_head', $plugin_public, 'add_manifest_link' );
        $this->loader->add_action( 'wp_footer', $plugin_public, 'add_subscription_dialog' );
        
        // Register REST API endpoints
        $plugin_api = new Quick_Push_Notifications_API( $this->get_plugin_name(), $this->get_version() );
        $this->loader->add_action( 'rest_api_init', $plugin_api, 'register_routes' );
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    Quick_Push_Notifications_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }
}
