<?php
/**
 * Handles subscriber management
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 */

/**
 * Handles subscriber management.
 *
 * This class handles the creation, retrieval, and management of push notification subscribers.
 *
 * @since      1.0.0
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 * @author     Quick Plugins <info@example.com>
 */
class Quick_Push_Notifications_Subscriber {

    /**
     * Add a new subscriber.
     *
     * @since    1.0.0
     * @param    array    $subscription_data    The subscription data from the browser.
     * @return   int|false                      The subscriber ID if successful, false otherwise.
     */
    public static function add_subscriber($subscription_data) {
        global $wpdb;
        
        if (empty($subscription_data) || !isset($subscription_data['endpoint'])) {
            return false;
        }
        
        $table_name = $wpdb->prefix . 'qpn_subscribers';
        $endpoint = sanitize_text_field($subscription_data['endpoint']);
        
        // Check if subscriber already exists
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table_name WHERE endpoint = %s",
            $endpoint
        ));
        
        if ($existing) {
            // Update the existing subscriber
            $wpdb->update(
                $table_name,
                array(
                    'subscription_data' => json_encode($subscription_data),
                    'last_activity' => current_time('mysql')
                ),
                array('endpoint' => $endpoint),
                array('%s', '%s'),
                array('%s')
            );
            
            return (int) $existing;
        }
        
        // Extract keys from subscription
        $auth = '';
        $p256dh = '';
        
        if (isset($subscription_data['keys'])) {
            $auth = isset($subscription_data['keys']['auth']) ? sanitize_text_field($subscription_data['keys']['auth']) : '';
            $p256dh = isset($subscription_data['keys']['p256dh']) ? sanitize_text_field($subscription_data['keys']['p256dh']) : '';
        }
        
        // Get user information
        $user_id = get_current_user_id();
        $ip_address = self::get_client_ip();
        
        // Get browser and OS information
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
        $browser = self::get_browser_name($user_agent);
        $os = self::get_os_name($user_agent);
        $device_type = self::get_device_type($user_agent);
        
        // Insert new subscriber
        $wpdb->insert(
            $table_name,
            array(
                'endpoint' => $endpoint,
                'auth_token' => $auth,
                'public_key' => $p256dh,
                'subscription_data' => json_encode($subscription_data),
                'user_id' => $user_id ? $user_id : null,
                'browser' => $browser,
                'os' => $os,
                'device_type' => $device_type,
                'ip_address' => $ip_address,
                'created_at' => current_time('mysql'),
                'last_activity' => current_time('mysql')
            ),
            array('%s', '%s', '%s', '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s')
        );
        
        $subscriber_id = $wpdb->insert_id;
        
        // Send welcome notification if enabled
        if (get_option('qpn_welcome_notification_enabled', 'yes') === 'yes') {
            $title = get_option('qpn_welcome_notification_title', 'Thanks for subscribing!');
            $message = get_option('qpn_welcome_notification_message', 'You will now receive notifications from our site.');
            $icon = get_option('qpn_notification_icon', '');
            
            $notification = array(
                'title' => $title,
                'message' => $message,
                'icon' => $icon,
                'url' => home_url()
            );
            
            // Send to this subscriber only
            Quick_Push_Notifications_Firebase::send_notification_to_subscribers(array($subscriber_id), $notification);
        }
        
        return $subscriber_id;
    }
    
    /**
     * Get all subscribers.
     *
     * @since    1.0.0
     * @param    array    $args    Query arguments.
     * @return   array             Array of subscribers.
     */
    public static function get_subscribers($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'limit' => 100,
            'offset' => 0,
            'orderby' => 'created_at',
            'order' => 'DESC',
            'search' => '',
            'browser' => '',
            'os' => '',
            'device_type' => ''
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $table_name = $wpdb->prefix . 'qpn_subscribers';
        
        $query = "SELECT * FROM $table_name WHERE 1=1";
        $prepare_args = array();
        
        // Add search condition
        if (!empty($args['search'])) {
            $query .= " AND (endpoint LIKE %s OR ip_address LIKE %s OR country LIKE %s OR city LIKE %s)";
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $prepare_args[] = $search_term;
            $prepare_args[] = $search_term;
            $prepare_args[] = $search_term;
            $prepare_args[] = $search_term;
        }
        
        // Add browser filter
        if (!empty($args['browser'])) {
            $query .= " AND browser = %s";
            $prepare_args[] = $args['browser'];
        }
        
        // Add OS filter
        if (!empty($args['os'])) {
            $query .= " AND os = %s";
            $prepare_args[] = $args['os'];
        }
        
        // Add device type filter
        if (!empty($args['device_type'])) {
            $query .= " AND device_type = %s";
            $prepare_args[] = $args['device_type'];
        }
        
        // Add order
        $query .= " ORDER BY " . sanitize_sql_orderby($args['orderby'] . ' ' . $args['order']);
        
        // Add limit
        $query .= " LIMIT %d OFFSET %d";
        $prepare_args[] = (int) $args['limit'];
        $prepare_args[] = (int) $args['offset'];
        
        // Prepare and execute the query
        $query = $wpdb->prepare($query, $prepare_args);
        $results = $wpdb->get_results($query, ARRAY_A);
        
        return $results;
    }
    
    /**
     * Count subscribers.
     *
     * @since    1.0.0
     * @param    array    $args    Query arguments.
     * @return   int               Count of subscribers.
     */
    public static function count_subscribers($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'search' => '',
            'browser' => '',
            'os' => '',
            'device_type' => ''
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $table_name = $wpdb->prefix . 'qpn_subscribers';
        
        $query = "SELECT COUNT(*) FROM $table_name WHERE 1=1";
        $prepare_args = array();
        
        // Add search condition
        if (!empty($args['search'])) {
            $query .= " AND (endpoint LIKE %s OR ip_address LIKE %s OR country LIKE %s OR city LIKE %s)";
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $prepare_args[] = $search_term;
            $prepare_args[] = $search_term;
            $prepare_args[] = $search_term;
            $prepare_args[] = $search_term;
        }
        
        // Add browser filter
        if (!empty($args['browser'])) {
            $query .= " AND browser = %s";
            $prepare_args[] = $args['browser'];
        }
        
        // Add OS filter
        if (!empty($args['os'])) {
            $query .= " AND os = %s";
            $prepare_args[] = $args['os'];
        }
        
        // Add device type filter
        if (!empty($args['device_type'])) {
            $query .= " AND device_type = %s";
            $prepare_args[] = $args['device_type'];
        }
        
        // Prepare and execute the query
        $query = $wpdb->prepare($query, $prepare_args);
        $count = $wpdb->get_var($query);
        
        return (int) $count;
    }
    
    /**
     * Delete a subscriber.
     *
     * @since    1.0.0
     * @param    int      $subscriber_id    The subscriber ID.
     * @return   bool                       True if successful, false otherwise.
     */
    public static function delete_subscriber($subscriber_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qpn_subscribers';
        
        $result = $wpdb->delete(
            $table_name,
            array('id' => $subscriber_id),
            array('%d')
        );
        
        return $result !== false;
    }
    
    /**
     * Get the client IP address.
     *
     * @since    1.0.0
     * @return   string    The client IP address.
     */
    private static function get_client_ip() {
        $ip_address = '';
        
        // Check for shared internet/ISP IP
        if (!empty($_SERVER['HTTP_CLIENT_IP']) && filter_var($_SERVER['HTTP_CLIENT_IP'], FILTER_VALIDATE_IP)) {
            $ip_address = $_SERVER['HTTP_CLIENT_IP'];
        }
        
        // Check for IPs passing through proxies
        elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            // Check if multiple IPs exist in X_FORWARDED_FOR
            $ip_list = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
            foreach ($ip_list as $ip) {
                if (filter_var(trim($ip), FILTER_VALIDATE_IP)) {
                    $ip_address = trim($ip);
                    break;
                }
            }
        }
        
        // If no valid IP was found, use REMOTE_ADDR
        elseif (!empty($_SERVER['REMOTE_ADDR']) && filter_var($_SERVER['REMOTE_ADDR'], FILTER_VALIDATE_IP)) {
            $ip_address = $_SERVER['REMOTE_ADDR'];
        }
        
        return sanitize_text_field($ip_address);
    }
    
    /**
     * Get the browser name from user agent.
     *
     * @since    1.0.0
     * @param    string    $user_agent    The user agent string.
     * @return   string                   The browser name.
     */
    private static function get_browser_name($user_agent) {
        if (strpos($user_agent, 'Opera') || strpos($user_agent, 'OPR/')) {
            return 'Opera';
        } elseif (strpos($user_agent, 'Edge')) {
            return 'Edge';
        } elseif (strpos($user_agent, 'Chrome')) {
            return 'Chrome';
        } elseif (strpos($user_agent, 'Safari')) {
            return 'Safari';
        } elseif (strpos($user_agent, 'Firefox')) {
            return 'Firefox';
        } elseif (strpos($user_agent, 'MSIE') || strpos($user_agent, 'Trident/7')) {
            return 'Internet Explorer';
        }
        
        return 'Unknown';
    }
    
    /**
     * Get the OS name from user agent.
     *
     * @since    1.0.0
     * @param    string    $user_agent    The user agent string.
     * @return   string                   The OS name.
     */
    private static function get_os_name($user_agent) {
        if (strpos($user_agent, 'Windows NT 10.0')) {
            return 'Windows 10';
        } elseif (strpos($user_agent, 'Windows NT 6.3')) {
            return 'Windows 8.1';
        } elseif (strpos($user_agent, 'Windows NT 6.2')) {
            return 'Windows 8';
        } elseif (strpos($user_agent, 'Windows NT 6.1')) {
            return 'Windows 7';
        } elseif (strpos($user_agent, 'Windows NT 6.0')) {
            return 'Windows Vista';
        } elseif (strpos($user_agent, 'Windows NT 5.1')) {
            return 'Windows XP';
        } elseif (strpos($user_agent, 'Windows NT 5.0')) {
            return 'Windows 2000';
        } elseif (strpos($user_agent, 'Mac')) {
            return 'Mac OS';
        } elseif (strpos($user_agent, 'X11')) {
            return 'UNIX';
        } elseif (strpos($user_agent, 'Linux')) {
            return 'Linux';
        } elseif (strpos($user_agent, 'Android')) {
            return 'Android';
        } elseif (strpos($user_agent, 'iPhone') || strpos($user_agent, 'iPad') || strpos($user_agent, 'iPod')) {
            return 'iOS';
        }
        
        return 'Unknown';
    }
    
    /**
     * Get the device type from user agent.
     *
     * @since    1.0.0
     * @param    string    $user_agent    The user agent string.
     * @return   string                   The device type.
     */
    private static function get_device_type($user_agent) {
        if (strpos($user_agent, 'Mobile') !== false || strpos($user_agent, 'Android') !== false && strpos($user_agent, 'Mobile') !== false) {
            return 'Mobile';
        } elseif (strpos($user_agent, 'iPad') !== false || strpos($user_agent, 'Android') !== false && strpos($user_agent, 'Mobile') === false) {
            return 'Tablet';
        } else {
            return 'Desktop';
        }
    }
}
