<?php
/**
 * Handles notification management
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 */

/**
 * Handles notification management.
 *
 * This class handles the creation, retrieval, and management of push notifications.
 *
 * @since      1.0.0
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 * @author     Quick Plugins <info@example.com>
 */
class Quick_Push_Notifications_Notification {

    /**
     * Save a notification.
     *
     * @since    1.0.0
     * @param    array    $notification_data    The notification data.
     * @return   int|false                      The notification ID if successful, false otherwise.
     */
    public static function save_notification($notification_data) {
        global $wpdb;
        
        if (empty($notification_data) || !isset($notification_data['title']) || !isset($notification_data['message'])) {
            return false;
        }
        
        $table_name = $wpdb->prefix . 'qpn_notifications';
        
        $data = array(
            'title' => sanitize_text_field($notification_data['title']),
            'message' => wp_kses_post($notification_data['message']),
            'url' => isset($notification_data['url']) ? esc_url_raw($notification_data['url']) : '',
            'icon' => isset($notification_data['icon']) ? esc_url_raw($notification_data['icon']) : '',
            'image' => isset($notification_data['image']) ? esc_url_raw($notification_data['image']) : '',
            'status' => isset($notification_data['status']) ? sanitize_text_field($notification_data['status']) : 'draft',
            'scheduled_at' => isset($notification_data['schedule_time']) ? sanitize_text_field($notification_data['schedule_time']) : null,
            'target_audience' => isset($notification_data['target_audience']) ? sanitize_text_field($notification_data['target_audience']) : 'all',
            'created_by' => get_current_user_id(),
            'updated_at' => current_time('mysql')
        );
        
        $format = array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s');
        
        // If notification ID is provided, update existing notification
        if (isset($notification_data['id']) && $notification_data['id'] > 0) {
            $wpdb->update(
                $table_name,
                $data,
                array('id' => $notification_data['id']),
                $format,
                array('%d')
            );
            
            return (int) $notification_data['id'];
        }
        
        // Add created_at for new notifications
        $data['created_at'] = current_time('mysql');
        $format[] = '%s';
        
        // Insert new notification
        $wpdb->insert(
            $table_name,
            $data,
            $format
        );
        
        $notification_id = $wpdb->insert_id;
        
        // If notification is scheduled, ensure the cron job is set up
        if ($data['status'] === 'scheduled' && !empty($data['scheduled_at'])) {
            self::ensure_scheduled_notification_cron();
        }
        
        return $notification_id;
    }
    
    /**
     * Get notifications.
     *
     * @since    1.0.0
     * @param    array    $args    Query arguments.
     * @return   array             Array of notifications.
     */
    public static function get_notifications($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'limit' => 100,
            'offset' => 0,
            'orderby' => 'created_at',
            'order' => 'DESC',
            'status' => '',
            'search' => ''
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $table_name = $wpdb->prefix . 'qpn_notifications';
        
        $query = "SELECT * FROM $table_name WHERE 1=1";
        $prepare_args = array();
        
        // Add status filter
        if (!empty($args['status'])) {
            $query .= " AND status = %s";
            $prepare_args[] = $args['status'];
        }
        
        // Add search condition
        if (!empty($args['search'])) {
            $query .= " AND (title LIKE %s OR message LIKE %s)";
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $prepare_args[] = $search_term;
            $prepare_args[] = $search_term;
        }
        
        // Add order
        $query .= " ORDER BY " . sanitize_sql_orderby($args['orderby'] . ' ' . $args['order']);
        
        // Add limit
        $query .= " LIMIT %d OFFSET %d";
        $prepare_args[] = (int) $args['limit'];
        $prepare_args[] = (int) $args['offset'];
        
        // Prepare and execute the query
        $query = $wpdb->prepare($query, $prepare_args);
        $results = $wpdb->get_results($query, ARRAY_A);
        
        return $results;
    }
    
    /**
     * Get a notification by ID.
     *
     * @since    1.0.0
     * @param    int      $notification_id    The notification ID.
     * @return   array|false                  The notification data if found, false otherwise.
     */
    public static function get_notification($notification_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qpn_notifications';
        
        $notification = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM $table_name WHERE id = %d",
                $notification_id
            ),
            ARRAY_A
        );
        
        return $notification;
    }
    
    /**
     * Delete a notification.
     *
     * @since    1.0.0
     * @param    int      $notification_id    The notification ID.
     * @return   bool                         True if successful, false otherwise.
     */
    public static function delete_notification($notification_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qpn_notifications';
        
        $result = $wpdb->delete(
            $table_name,
            array('id' => $notification_id),
            array('%d')
        );
        
        return $result !== false;
    }
    
    /**
     * Send a notification.
     *
     * @since    1.0.0
     * @param    int      $notification_id    The notification ID.
     * @return   array                        The result of the send operation.
     */
    public static function send_notification($notification_id) {
        global $wpdb;
        
        $notification = self::get_notification($notification_id);
        
        if (!$notification) {
            return array('error' => 'Notification not found');
        }
        
        // Prepare notification data
        $notification_data = array(
            'id' => $notification['id'],
            'title' => $notification['title'],
            'message' => $notification['message'],
            'url' => $notification['url'],
            'icon' => $notification['icon'],
            'image' => $notification['image']
        );
        
        // Determine target audience
        $target_audience = $notification['target_audience'];
        
        if ($target_audience === 'all') {
            // Send to all subscribers
            $result = Quick_Push_Notifications_Firebase::send_notification_to_all($notification_data);
        } else {
            // TODO: Implement segmentation for different target audiences
            $result = Quick_Push_Notifications_Firebase::send_notification_to_all($notification_data);
        }
        
        // Update notification status
        $wpdb->update(
            $wpdb->prefix . 'qpn_notifications',
            array(
                'status' => 'sent',
                'sent_at' => current_time('mysql'),
                'updated_at' => current_time('mysql')
            ),
            array('id' => $notification_id),
            array('%s', '%s', '%s'),
            array('%d')
        );
        
        return $result;
    }
    
    /**
     * Get scheduled notifications that are due to be sent.
     *
     * @since    1.0.0
     * @return   array    Array of notifications due to be sent.
     */
    public static function get_due_scheduled_notifications() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'qpn_notifications';
        
        $current_time = current_time('mysql');
        
        $notifications = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table_name WHERE status = 'scheduled' AND scheduled_at <= %s",
                $current_time
            ),
            ARRAY_A
        );
        
        return $notifications;
    }
    
    /**
     * Process scheduled notifications.
     *
     * @since    1.0.0
     */
    public static function process_scheduled_notifications() {
        $notifications = self::get_due_scheduled_notifications();
        
        if (empty($notifications)) {
            return;
        }
        
        foreach ($notifications as $notification) {
            self::send_notification($notification['id']);
        }
    }
    
    /**
     * Ensure the scheduled notification cron job is set up.
     *
     * @since    1.0.0
     */
    private static function ensure_scheduled_notification_cron() {
        if (!wp_next_scheduled('qpn_send_scheduled_notifications')) {
            wp_schedule_event(time(), 'hourly', 'qpn_send_scheduled_notifications');
        }
    }
    
    /**
     * Create a notification for a new post.
     *
     * @since    1.0.0
     * @param    int      $post_id    The post ID.
     * @return   int|false            The notification ID if successful, false otherwise.
     */
    public static function create_post_notification($post_id) {
        $post = get_post($post_id);
        
        if (!$post || $post->post_status !== 'publish') {
            return false;
        }
        
        // Check if auto-push is enabled for this post type
        $post_types = get_option('qpn_auto_push_post_types', array('post'));
        
        if (!in_array($post->post_type, $post_types)) {
            return false;
        }
        
        // Get notification template
        $title_template = get_option('qpn_auto_push_title', 'New post: {post_title}');
        $message_template = get_option('qpn_auto_push_message', '{post_excerpt}');
        
        // Replace placeholders
        $title = str_replace(
            array('{post_title}', '{site_name}'),
            array($post->post_title, get_bloginfo('name')),
            $title_template
        );
        
        $excerpt = has_excerpt($post_id) ? get_the_excerpt($post_id) : wp_trim_words(strip_shortcodes(wp_strip_all_tags($post->post_content)), 55, '...');
        
        $message = str_replace(
            array('{post_title}', '{post_excerpt}', '{site_name}'),
            array($post->post_title, $excerpt, get_bloginfo('name')),
            $message_template
        );
        
        // Get featured image
        $image = '';
        if (has_post_thumbnail($post_id)) {
            $image_id = get_post_thumbnail_id($post_id);
            $image_url = wp_get_attachment_image_src($image_id, 'large');
            if ($image_url) {
                $image = $image_url[0];
            }
        }
        
        // Create notification
        $notification_data = array(
            'title' => $title,
            'message' => $message,
            'url' => get_permalink($post_id),
            'icon' => get_option('qpn_notification_icon', ''),
            'image' => $image,
            'status' => 'draft'
        );
        
        // Save notification
        $notification_id = self::save_notification($notification_data);
        
        // If auto-send is enabled, send the notification
        if (get_option('qpn_auto_push_send', 'no') === 'yes') {
            self::send_notification($notification_id);
        }
        
        return $notification_id;
    }
}
