<?php
/**
 * Handles API endpoints
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 */

/**
 * Handles API endpoints.
 *
 * This class handles the REST API endpoints for the plugin.
 *
 * @since      1.0.0
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 * @author     Quick Plugins <info@example.com>
 */
class Quick_Push_Notifications_API {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the REST API routes.
     *
     * @since    1.0.0
     */
    public function register_routes() {
        register_rest_route('quick-push-notifications/v1', '/subscribe', array(
            'methods' => 'POST',
            'callback' => array($this, 'subscribe'),
            'permission_callback' => '__return_true'
        ));

        register_rest_route('quick-push-notifications/v1', '/unsubscribe', array(
            'methods' => 'POST',
            'callback' => array($this, 'unsubscribe'),
            'permission_callback' => '__return_true'
        ));

        register_rest_route('quick-push-notifications/v1', '/firebase-config', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_firebase_config'),
            'permission_callback' => '__return_true'
        ));

        register_rest_route('quick-push-notifications/v1', '/manifest.json', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_manifest'),
            'permission_callback' => '__return_true'
        ));
    }

    /**
     * Handle subscription requests.
     *
     * @since    1.0.0
     * @param    WP_REST_Request    $request    The request object.
     * @return   WP_REST_Response               The response object.
     */
    public function subscribe($request) {
        $params = $request->get_params();
        
        if (!isset($params['subscription']) || empty($params['subscription'])) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'Subscription data is required'
            ), 400);
        }
        
        $subscription_data = $params['subscription'];
        
        // Add subscriber
        $subscriber_id = Quick_Push_Notifications_Subscriber::add_subscriber($subscription_data);
        
        if (!$subscriber_id) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'Failed to add subscriber'
            ), 500);
        }
        
        return new WP_REST_Response(array(
            'success' => true,
            'subscriber_id' => $subscriber_id,
            'message' => 'Successfully subscribed'
        ), 200);
    }

    /**
     * Handle unsubscription requests.
     *
     * @since    1.0.0
     * @param    WP_REST_Request    $request    The request object.
     * @return   WP_REST_Response               The response object.
     */
    public function unsubscribe($request) {
        $params = $request->get_params();
        
        if (!isset($params['endpoint']) || empty($params['endpoint'])) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'Endpoint is required'
            ), 400);
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'qpn_subscribers';
        
        // Find subscriber by endpoint
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $table_name WHERE endpoint = %s",
            $params['endpoint']
        ));
        
        if (!$subscriber) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'Subscriber not found'
            ), 404);
        }
        
        // Delete subscriber
        $result = Quick_Push_Notifications_Subscriber::delete_subscriber($subscriber->id);
        
        if (!$result) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'Failed to unsubscribe'
            ), 500);
        }
        
        return new WP_REST_Response(array(
            'success' => true,
            'message' => 'Successfully unsubscribed'
        ), 200);
    }

    /**
     * Get Firebase configuration.
     *
     * @since    1.0.0
     * @param    WP_REST_Request    $request    The request object.
     * @return   WP_REST_Response               The response object.
     */
    public function get_firebase_config($request) {
        $config = Quick_Push_Notifications_Firebase::get_firebase_config();
        
        return new WP_REST_Response($config, 200);
    }

    /**
     * Get web app manifest.
     *
     * @since    1.0.0
     * @param    WP_REST_Request    $request    The request object.
     * @return   WP_REST_Response               The response object.
     */
    public function get_manifest($request) {
        $site_name = get_bloginfo('name');
        $site_url = home_url();
        $icon = get_option('qpn_notification_icon', '');
        
        if (empty($icon)) {
            // Try to use site icon if available
            $site_icon_id = get_option('site_icon');
            if ($site_icon_id) {
                $icon_url = wp_get_attachment_image_src($site_icon_id, 'full');
                if ($icon_url) {
                    $icon = $icon_url[0];
                }
            }
        }
        
        $manifest = array(
            'name' => $site_name,
            'short_name' => $site_name,
            'start_url' => $site_url,
            'display' => 'standalone',
            'background_color' => '#ffffff',
            'theme_color' => '#000000',
            'gcm_sender_id' => get_option('qpn_firebase_messaging_sender_id', ''),
            'icons' => array(
                array(
                    'src' => $icon,
                    'sizes' => '192x192',
                    'type' => 'image/png'
                ),
                array(
                    'src' => $icon,
                    'sizes' => '512x512',
                    'type' => 'image/png'
                )
            )
        );
        
        return new WP_REST_Response($manifest, 200);
    }
}
