<?php
/**
 * Fired during plugin activation
 *
 * @link       https://plugins.quickpr.co/
 * @since      1.0.0
 *
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 */

/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since      1.0.0
 * @package    Quick_Push_Notifications
 * @subpackage Quick_Push_Notifications/includes
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Push_Notifications_Activator {

    /**
     * Activate the plugin.
     *
     * Create necessary database tables and set up default options.
     *
     * @since    1.0.0
     */
    public static function activate() {
        global $wpdb;
        
        // Create database tables
        $charset_collate = $wpdb->get_charset_collate();
        
        // Table for subscribers
        $table_name = $wpdb->prefix . 'qpn_subscribers';
        $sql = "CREATE TABLE $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            endpoint varchar(255) NOT NULL,
            auth_token varchar(100) NOT NULL,
            public_key varchar(100) NOT NULL,
            subscription_data text NOT NULL,
            user_id bigint(20) DEFAULT NULL,
            browser varchar(50) DEFAULT NULL,
            os varchar(50) DEFAULT NULL,
            device_type varchar(20) DEFAULT NULL,
            ip_address varchar(45) DEFAULT NULL,
            country varchar(50) DEFAULT NULL,
            city varchar(50) DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            last_activity datetime DEFAULT NULL,
            PRIMARY KEY  (id),
            UNIQUE KEY endpoint (endpoint),
            KEY user_id (user_id)
        ) $charset_collate;";
        
        // Table for notifications
        $table_name_notifications = $wpdb->prefix . 'qpn_notifications';
        $sql .= "CREATE TABLE $table_name_notifications (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            title varchar(255) NOT NULL,
            message text NOT NULL,
            url varchar(255) DEFAULT NULL,
            icon varchar(255) DEFAULT NULL,
            image varchar(255) DEFAULT NULL,
            segments text DEFAULT NULL,
            scheduled_at datetime DEFAULT NULL,
            status varchar(20) DEFAULT 'draft',
            sent_at datetime DEFAULT NULL,
            created_by bigint(20) DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT NULL,
            PRIMARY KEY  (id),
            KEY status (status),
            KEY scheduled_at (scheduled_at)
        ) $charset_collate;";
        
        // Table for notification logs
        $table_name_logs = $wpdb->prefix . 'qpn_notification_logs';
        $sql .= "CREATE TABLE $table_name_logs (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            notification_id bigint(20) NOT NULL,
            subscriber_id bigint(20) NOT NULL,
            status varchar(20) DEFAULT 'sent',
            delivered_at datetime DEFAULT NULL,
            clicked_at datetime DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY  (id),
            KEY notification_id (notification_id),
            KEY subscriber_id (subscriber_id)
        ) $charset_collate;";
        
        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql );
        
        // Set default options
        $default_options = array(
            'firebase_api_key' => '',
            'firebase_auth_domain' => '',
            'firebase_project_id' => '',
            'firebase_storage_bucket' => '',
            'firebase_messaging_sender_id' => '',
            'firebase_app_id' => '',
            'firebase_server_key' => '',
            'subscription_dialog_title' => 'Subscribe to Notifications',
            'subscription_dialog_text' => 'Get notified about the latest updates and news.',
            'subscription_dialog_allow_button' => 'Allow',
            'subscription_dialog_cancel_button' => 'Not Now',
            'subscription_dialog_delay' => '5',
            'subscription_dialog_show_on' => 'all',
            'notification_icon' => '',
            'welcome_notification_enabled' => 'yes',
            'welcome_notification_title' => 'Thanks for subscribing!',
            'welcome_notification_message' => 'You will now receive notifications from our site.',
            'auto_push_new_post' => 'no',
            'auto_push_post_types' => array('post'),
            'auto_push_post_title' => 'New Post: {post_title}',
            'auto_push_post_message' => '{post_excerpt}',
        );
        
        foreach ( $default_options as $option_name => $option_value ) {
            add_option( 'qpn_' . $option_name, $option_value );
        }
        
        // Create a directory for service worker files
        $upload_dir = wp_upload_dir();
        $qpn_dir = $upload_dir['basedir'] . '/quick-push-notifications';
        
        if (!file_exists($qpn_dir)) {
            wp_mkdir_p($qpn_dir);
        }
        
        // Create an empty index.php file to prevent directory listing
        if (!file_exists($qpn_dir . '/index.php')) {
            $index_file = fopen($qpn_dir . '/index.php', 'w');
            fwrite($index_file, '<?php // Silence is golden');
            fclose($index_file);
        }
        
        // Add a cron job for scheduled notifications
        if (!wp_next_scheduled('qpn_send_scheduled_notifications')) {
            wp_schedule_event(time(), 'hourly', 'qpn_send_scheduled_notifications');
        }
    }
}
