<?php
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://plugins.quickpr.co/
 * @since      1.0.0
 *
 * @package    Quick_Custom_Admin
 * @subpackage Quick_Custom_Admin/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Quick_Custom_Admin
 * @subpackage Quick_Custom_Admin/includes
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Custom_Admin {

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Quick_Custom_Admin_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if (defined('QUICK_CUSTOM_ADMIN_VERSION')) {
            $this->version = QUICK_CUSTOM_ADMIN_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'quick-custom-admin';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - Quick_Custom_Admin_Loader. Orchestrates the hooks of the plugin.
     * - Quick_Custom_Admin_i18n. Defines internationalization functionality.
     * - Quick_Custom_Admin_Admin. Defines all hooks for the admin area.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-quick-custom-admin-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-quick-custom-admin-i18n.php';

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-quick-custom-admin-admin.php';

        $this->loader = new Quick_Custom_Admin_Loader();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Quick_Custom_Admin_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {
        $plugin_i18n = new Quick_Custom_Admin_i18n();
        $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {
        $plugin_admin = new Quick_Custom_Admin_Admin($this->get_plugin_name(), $this->get_version());

        // Enqueue styles and scripts
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');
        
        // Add menu items
        $this->loader->add_action('admin_menu', $plugin_admin, 'add_plugin_admin_menu');
        
        // Add Settings link to the plugin
        $this->loader->add_filter('plugin_action_links_' . plugin_basename(plugin_dir_path(dirname(__FILE__)) . $this->plugin_name . '.php'), $plugin_admin, 'add_action_links');
        
        // Register settings
        $this->loader->add_action('admin_init', $plugin_admin, 'register_settings');
        
        // White labeling hooks
        $this->loader->add_filter('admin_title', $plugin_admin, 'custom_admin_title', 10, 2);
        $this->loader->add_action('login_enqueue_scripts', $plugin_admin, 'custom_login_logo');
        $this->loader->add_filter('login_headerurl', $plugin_admin, 'custom_login_logo_url');
        $this->loader->add_filter('login_headertext', $plugin_admin, 'custom_login_logo_title');
        $this->loader->add_action('admin_bar_menu', $plugin_admin, 'replace_wp_logo', 999);
        $this->loader->add_filter('admin_footer_text', $plugin_admin, 'custom_admin_footer_text');
        $this->loader->add_filter('update_footer', $plugin_admin, 'custom_admin_version_footer_text', 999);
        
        // Dashboard customization
        $this->loader->add_action('wp_dashboard_setup', $plugin_admin, 'remove_default_dashboard_widgets');
        $this->loader->add_action('wp_dashboard_setup', $plugin_admin, 'add_custom_dashboard_widgets');
        
        // Admin menu customization
        $this->loader->add_action('admin_menu', $plugin_admin, 'customize_admin_menu', 999);
        $this->loader->add_action('admin_head', $plugin_admin, 'hide_admin_menu_items_css');
        
        // Role-based restrictions
        $this->loader->add_action('current_screen', $plugin_admin, 'restrict_admin_pages');
        $this->loader->add_filter('user_has_cap', $plugin_admin, 'adjust_user_capabilities', 10, 3);
        
        // Admin color schemes
        $this->loader->add_action('admin_init', $plugin_admin, 'register_custom_color_schemes');
        
        // Admin notices
        $this->loader->add_action('admin_notices', $plugin_admin, 'display_custom_admin_notices');
        $this->loader->add_action('admin_init', $plugin_admin, 'dismiss_admin_notices');
        
        // Help tabs
        $this->loader->add_action('admin_head', $plugin_admin, 'add_custom_help_tabs');
        
        // Admin columns
        $this->loader->add_filter('manage_posts_columns', $plugin_admin, 'customize_post_columns');
        $this->loader->add_action('manage_posts_custom_column', $plugin_admin, 'populate_custom_columns', 10, 2);
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    Quick_Custom_Admin_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }
}
