<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://plugins.quickpr.co/
 * @since      1.0.0
 *
 * @package    Quick_Custom_Admin
 * @subpackage Quick_Custom_Admin/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Quick_Custom_Admin
 * @subpackage Quick_Custom_Admin/admin
 * @author     Quick LLC <info@quickpr.co>
 */
class Quick_Custom_Admin_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * The options of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $options    The options of this plugin.
     */
    private $options;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->options = get_option('quick_custom_admin_options', array());
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/quick-custom-admin-admin.css', array(), $this->version, 'all');
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        wp_enqueue_media();
        wp_enqueue_script('wp-color-picker');
        wp_enqueue_script('jquery-ui-sortable');
        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/quick-custom-admin-admin.js', array('jquery', 'wp-color-picker', 'jquery-ui-sortable'), $this->version, false);
        
        wp_localize_script($this->plugin_name, 'quick_custom_admin_params', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('quick-custom-admin-nonce'),
            'i18n' => array(
                'select_image' => __('Select Image', 'quick-custom-admin'),
                'use_image' => __('Use This Image', 'quick-custom-admin'),
                'remove_image' => __('Remove Image', 'quick-custom-admin'),
                'confirm_delete' => __('Are you sure you want to delete this item? This action cannot be undone.', 'quick-custom-admin'),
            ),
        ));
    }

    /**
     * Add menu items.
     *
     * @since    1.0.0
     */
    public function add_plugin_admin_menu() {
        // Main menu item
        add_menu_page(
            __('Quick Custom Admin', 'quick-custom-admin'),
            __('Custom Admin', 'quick-custom-admin'),
            'manage_options',
            $this->plugin_name,
            array($this, 'display_plugin_admin_page'),
            'dashicons-admin-customizer',
            81
        );
        
        // White Label submenu
        add_submenu_page(
            $this->plugin_name,
            __('White Label', 'quick-custom-admin'),
            __('White Label', 'quick-custom-admin'),
            'manage_options',
            $this->plugin_name,
            array($this, 'display_plugin_admin_page')
        );
        
        // Dashboard submenu
        add_submenu_page(
            $this->plugin_name,
            __('Dashboard', 'quick-custom-admin'),
            __('Dashboard', 'quick-custom-admin'),
            'manage_options',
            $this->plugin_name . '-dashboard',
            array($this, 'display_plugin_dashboard_page')
        );
        
        // Admin Menu submenu
        add_submenu_page(
            $this->plugin_name,
            __('Admin Menu', 'quick-custom-admin'),
            __('Admin Menu', 'quick-custom-admin'),
            'manage_options',
            $this->plugin_name . '-admin-menu',
            array($this, 'display_plugin_admin_menu_page')
        );
        
        // Role Restrictions submenu
        add_submenu_page(
            $this->plugin_name,
            __('Role Restrictions', 'quick-custom-admin'),
            __('Role Restrictions', 'quick-custom-admin'),
            'manage_options',
            $this->plugin_name . '-role-restrictions',
            array($this, 'display_plugin_role_restrictions_page')
        );
        
        // Admin Columns submenu
        add_submenu_page(
            $this->plugin_name,
            __('Admin Columns', 'quick-custom-admin'),
            __('Admin Columns', 'quick-custom-admin'),
            'manage_options',
            $this->plugin_name . '-admin-columns',
            array($this, 'display_plugin_admin_columns_page')
        );
        
        // Admin Notices submenu
        add_submenu_page(
            $this->plugin_name,
            __('Admin Notices', 'quick-custom-admin'),
            __('Admin Notices', 'quick-custom-admin'),
            'manage_options',
            $this->plugin_name . '-admin-notices',
            array($this, 'display_plugin_admin_notices_page')
        );
    }

    /**
     * Add action links to the plugin list table.
     *
     * @since    1.0.0
     * @param    array    $links    An array of plugin action links.
     * @return   array              An array of plugin action links.
     */
    public function add_action_links($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=' . $this->plugin_name) . '">' . __('Settings', 'quick-custom-admin') . '</a>';
        
        // Add the settings link before other links
        array_unshift($links, $settings_link);
        
        return $links;
    }

    /**
     * Register plugin settings.
     *
     * @since    1.0.0
     */
    public function register_settings() {
        register_setting(
            'quick_custom_admin_options',
            'quick_custom_admin_options',
            array($this, 'validate_options')
        );
    }

    /**
     * Validate plugin options.
     *
     * @since    1.0.0
     * @param    array    $input    The options to validate.
     * @return   array              The validated options.
     */
    public function validate_options($input) {
        // Implement validation logic here
        return $input;
    }

    /**
     * Render the settings page for this plugin.
     *
     * @since    1.0.0
     */
    public function display_plugin_admin_page() {
        include_once 'partials/quick-custom-admin-admin-display.php';
    }

    /**
     * Render the dashboard settings page for this plugin.
     *
     * @since    1.0.0
     */
    public function display_plugin_dashboard_page() {
        include_once 'partials/quick-custom-admin-admin-dashboard.php';
    }

    /**
     * Render the admin menu settings page for this plugin.
     *
     * @since    1.0.0
     */
    public function display_plugin_admin_menu_page() {
        include_once 'partials/quick-custom-admin-admin-menu.php';
    }

    /**
     * Render the role restrictions settings page for this plugin.
     *
     * @since    1.0.0
     */
    public function display_plugin_role_restrictions_page() {
        include_once 'partials/quick-custom-admin-role-restrictions.php';
    }

    /**
     * Render the admin columns settings page for this plugin.
     *
     * @since    1.0.0
     */
    public function display_plugin_admin_columns_page() {
        include_once 'partials/quick-custom-admin-admin-columns.php';
    }

    /**
     * Render the admin notices settings page for this plugin.
     *
     * @since    1.0.0
     */
    public function display_plugin_admin_notices_page() {
        include_once 'partials/quick-custom-admin-admin-notices.php';
    }

    /**
     * Customize the admin title.
     *
     * @since    1.0.0
     * @param    string    $admin_title    The admin title.
     * @param    string    $title          The original title.
     * @return   string                    The customized admin title.
     */
    public function custom_admin_title($admin_title, $title) {
        if (!empty($this->options['white_label_enabled']) && !empty($this->options['custom_admin_title'])) {
            return str_replace('WordPress', $this->options['custom_admin_title'], $admin_title);
        }
        
        return $admin_title;
    }

    /**
     * Customize the login logo.
     *
     * @since    1.0.0
     */
    public function custom_login_logo() {
        if (!empty($this->options['white_label_enabled']) && !empty($this->options['custom_login_logo'])) {
            $logo_url = $this->options['custom_login_logo'];
            
            echo '<style type="text/css">
                #login h1 a, .login h1 a {
                    background-image: url(' . esc_url($logo_url) . ');
                    background-size: contain;
                    background-repeat: no-repeat;
                    background-position: center center;
                    width: 320px;
                    height: 80px;
                    margin: 0 auto;
                }
            </style>';
        }
    }

    /**
     * Customize the login logo URL.
     *
     * @since    1.0.0
     * @param    string    $url    The login logo URL.
     * @return   string            The customized login logo URL.
     */
    public function custom_login_logo_url($url) {
        if (!empty($this->options['white_label_enabled']) && !empty($this->options['custom_login_logo_url'])) {
            return esc_url($this->options['custom_login_logo_url']);
        }
        
        return $url;
    }

    /**
     * Customize the login logo title.
     *
     * @since    1.0.0
     * @param    string    $title    The login logo title.
     * @return   string              The customized login logo title.
     */
    public function custom_login_logo_title($title) {
        if (!empty($this->options['white_label_enabled']) && !empty($this->options['custom_login_logo_title'])) {
            return esc_html($this->options['custom_login_logo_title']);
        }
        
        return $title;
    }

    /**
     * Replace the WordPress logo in the admin bar.
     *
     * @since    1.0.0
     * @param    WP_Admin_Bar    $wp_admin_bar    The admin bar object.
     */
    public function replace_wp_logo($wp_admin_bar) {
        if (!empty($this->options['white_label_enabled']) && !empty($this->options['replace_wp_logo'])) {
            $wp_admin_bar->remove_node('wp-logo');
            
            if (!empty($this->options['custom_admin_logo'])) {
                $title = !empty($this->options['custom_login_logo_title']) ? $this->options['custom_login_logo_title'] : get_bloginfo('name');
                
                $wp_admin_bar->add_node(array(
                    'id'    => 'custom-logo',
                    'title' => '<img src="' . esc_url($this->options['custom_admin_logo']) . '" alt="' . esc_attr($title) . '" style="height: 20px; width: auto; padding-top: 6px;" />',
                    'href'  => !empty($this->options['custom_login_logo_url']) ? $this->options['custom_login_logo_url'] : home_url(),
                ));
            }
        }
    }

    /**
     * Customize the admin footer text.
     *
     * @since    1.0.0
     * @param    string    $text    The admin footer text.
     * @return   string             The customized admin footer text.
     */
    public function custom_admin_footer_text($text) {
        if (!empty($this->options['white_label_enabled']) && !empty($this->options['custom_admin_footer_text'])) {
            return wp_kses_post($this->options['custom_admin_footer_text']);
        }
        
        return $text;
    }

    /**
     * Customize the admin version footer text.
     *
     * @since    1.0.0
     * @param    string    $text    The admin version footer text.
     * @return   string             The customized admin version footer text.
     */
    public function custom_admin_version_footer_text($text) {
        if (!empty($this->options['white_label_enabled'])) {
            if (!empty($this->options['hide_wp_version'])) {
                return '';
            } elseif (!empty($this->options['custom_version_text'])) {
                return wp_kses_post($this->options['custom_version_text']);
            }
        }
        
        return $text;
    }
}
